<?php

namespace Drupal\wse_parallel\ValueObject;

/**
 * Data transfer object representing parallel editing state for an entity.
 *
 * This immutable value object contains information about parallel activity
 * including active sessions, publish history, and divergence status.
 *
 * @phpstan-type SessionArray array{sid: int, entity_type: string, entity_id: int, workspace_id: string, uid: int, started: int, last_seen: int}
 * @phpstan-type PublishArray array{plid: int, entity_type: string, entity_id: int, from_revision_id: int, to_revision_id: int, workspace_id: string, published: int, langcode: string|null}
 */
class ParallelStateDto {

  /**
   * Whether a newer publish exists since the base timestamp.
   *
   * @var bool
   */
  private $hasNewerPublish;

  /**
   * Active editing sessions.
   *
   * @var array<int, array<string, mixed>>
   * @phpstan-var array<int, SessionArray>
   */
  private $activeSessions;

  /**
   * Latest publish record.
   *
   * @var array<string, mixed>|null
   * @phpstan-var PublishArray|null
   */
  private $lastPublished;

  /**
   * Whether the entity has diverged from its base.
   *
   * @var bool
   */
  private $divergedFromBase;

  /**
   * Suggested action based on parallel state.
   *
   * @var string
   * @phpstan-var 'none'|'review'|'merge'|'conflict'
   */
  private $suggestedAction;

  /**
   * Constructs a ParallelStateDto.
   *
   * @param bool $hasNewerPublish
   *   Whether a newer publish exists.
   * @param array<int, array<string, mixed>> $activeSessions
   *   Active editing sessions.
   * @param array<string, mixed>|null $lastPublished
   *   Latest publish record.
   * @param bool $divergedFromBase
   *   Whether diverged from base.
   * @param string $suggestedAction
   *   Suggested action.
   *
   * @phpstan-param array<int, SessionArray> $activeSessions
   * @phpstan-param PublishArray|null $lastPublished
   * @phpstan-param 'none'|'review'|'merge'|'conflict' $suggestedAction
   */
  public function __construct(bool $hasNewerPublish, array $activeSessions, ?array $lastPublished, bool $divergedFromBase, string $suggestedAction) {
    $this->hasNewerPublish = $hasNewerPublish;
    $this->activeSessions = $activeSessions;
    $this->lastPublished = $lastPublished;
    $this->divergedFromBase = $divergedFromBase;
    $this->suggestedAction = $suggestedAction;
  }

  /**
   * Returns whether a newer publish exists.
   *
   * @return bool
   */
  public function hasNewerPublish(): bool {
    return $this->hasNewerPublish;
  }

  /**
   * Returns active editing sessions.
   *
   * @return array<int, array<string, mixed>>
   * @phpstan-return array<int, SessionArray>
   */
  public function getActiveSessions(): array {
    return $this->activeSessions;
  }

  /**
   * Returns the number of active sessions.
   *
   * @return int
   */
  public function getActiveSessionCount(): int {
    return count($this->activeSessions);
  }

  /**
   * Returns whether there are active sessions.
   *
   * @return bool
   */
  public function hasActiveSessions(): bool {
    return !empty($this->activeSessions);
  }

  /**
   * Returns the latest publish record.
   *
   * @return array<string, mixed>|null
   * @phpstan-return PublishArray|null
   */
  public function getLastPublished(): ?array {
    return $this->lastPublished;
  }

  /**
   * Returns whether the entity has been published.
   *
   * @return bool
   */
  public function hasBeenPublished(): bool {
    return $this->lastPublished !== NULL;
  }

  /**
   * Returns whether the entity has diverged from base.
   *
   * @return bool
   */
  public function hasDiverged(): bool {
    return $this->divergedFromBase;
  }

  /**
   * Returns the suggested action.
   *
   * @return string
   * @phpstan-return 'none'|'review'|'merge'|'conflict'
   */
  public function getSuggestedAction(): string {
    return $this->suggestedAction;
  }

  /**
   * Returns whether any parallel activity exists.
   *
   * @return bool
   */
  public function hasParallelActivity(): bool {
    return $this->hasNewerPublish
      || $this->hasActiveSessions()
      || $this->divergedFromBase;
  }

  /**
   * Returns a summary of the parallel state.
   *
   * @return array<string, mixed>
   *   Summary array with keys:
   *   - has_newer_publish: bool
   *   - active_session_count: int
   *   - has_diverged: bool
   *   - suggested_action: string
   *   - has_parallel_activity: bool
   */
  public function toArray(): array {
    return [
      'has_newer_publish' => $this->hasNewerPublish,
      'active_session_count' => $this->getActiveSessionCount(),
      'has_diverged' => $this->divergedFromBase,
      'suggested_action' => $this->suggestedAction,
      'has_parallel_activity' => $this->hasParallelActivity(),
      'last_published' => $this->lastPublished,
    ];
  }

}
