<?php

namespace Drupal\wse_parallel\Workspaces;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Utility\Error;
use Drupal\workspaces\Event\WorkspacePostPublishEvent;
use Drupal\workspaces\WorkspaceAssociationInterface;
use Drupal\workspaces\WorkspaceInterface;
use Drupal\workspaces\WorkspaceRepositoryInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Decorates the workspace association service to enable parallel editing.
 */
class ParallelWorkspaceAssociation implements WorkspaceAssociationInterface, EventSubscriberInterface {

  /**
   * Whether a workspace revert operation is currently in progress.
   *
   * @var bool
   */
  protected bool $isReverting = FALSE;

  /**
   * Constructs a ParallelWorkspaceAssociation object.
   *
   * @param \Drupal\workspaces\WorkspaceAssociationInterface $innerAssociation
   *   The decorated workspace association service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\workspaces\WorkspaceRepositoryInterface $workspaceRepository
   *   The workspace repository service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(protected WorkspaceAssociationInterface $innerAssociation, protected Connection $database, protected WorkspaceRepositoryInterface $workspaceRepository, protected LoggerInterface $logger) {}

  /**
   * Sets whether a revert operation is in progress.
   *
   * During revert operations, parallel editing bypass is disabled to ensure
   * accurate workspace tracking is maintained.
   *
   * @param bool $reverting
   *   TRUE if reverting, FALSE otherwise.
   */
  public function setReverting(bool $reverting): void {
    $this->isReverting = $reverting;
  }

  /**
   * {@inheritdoc}
   *
   * This is a copy of the core WorkspaceAssociation::trackEntity() method.
   * We maintain our own copy so that when it calls $this->getEntityTrackingWorkspaceIds(),
   * it will call OUR overridden version which respects the $isReverting flag,
   * rather than going into the inner service where the decorator chain breaks down.
   */
  public function trackEntity(RevisionableInterface $entity, WorkspaceInterface $workspace) {
    // Determine all workspaces that might be affected by this change.
    $affected_workspaces = $this->workspaceRepository->getDescendantsAndSelf($workspace->id());

    // Get the currently tracked revision for this workspace.
    $tracked = $this->getTrackedEntities($workspace->id(), $entity->getEntityTypeId(), [$entity->id()]);

    $tracked_revision_id = NULL;
    if (isset($tracked[$entity->getEntityTypeId()])) {
      $tracked_revision_id = key($tracked[$entity->getEntityTypeId()]);
    }

    try {
      $transaction = $this->database->startTransaction();
      // Update all affected workspaces that were tracking the current revision.
      // This means they are inheriting content and should be updated.
      if ($tracked_revision_id) {
        $this->database->update('workspace_association')
          ->fields([
            'target_entity_revision_id' => $entity->getRevisionId(),
          ])
          ->condition('workspace', $affected_workspaces, 'IN')
          ->condition('target_entity_type_id', $entity->getEntityTypeId())
          ->condition('target_entity_id', $entity->id())
          // Only update descendant workspaces if they have the same initial
          // revision, which means they are currently inheriting content.
          ->condition('target_entity_revision_id', $tracked_revision_id)
          ->execute();
      }

      // Insert a new index entry for each workspace that is not tracking this
      // entity yet.
      // Note: We query the database directly instead of using getEntityTrackingWorkspaceIds()
      // because that method may return an empty array during parallel editing operations,
      // which would incorrectly indicate that all workspaces are missing.
      $existing_workspaces = $this->database->select('workspace_association', 'wa')
        ->fields('wa', ['workspace'])
        ->condition('target_entity_type_id', $entity->getEntityTypeId())
        ->condition('target_entity_id', $entity->id())
        ->execute()
        ->fetchCol();

      $missing_workspaces = array_diff($affected_workspaces, $existing_workspaces);
      if ($missing_workspaces) {
        $insert_query = $this->database->insert('workspace_association')
          ->fields([
            'workspace',
            'target_entity_revision_id',
            'target_entity_type_id',
            'target_entity_id',
          ]);
        foreach ($missing_workspaces as $workspace_id) {
          $insert_query->values([
            'workspace' => $workspace_id,
            'target_entity_type_id' => $entity->getEntityTypeId(),
            'target_entity_id' => $entity->id(),
            'target_entity_revision_id' => $entity->getRevisionId(),
          ]);
        }
        $insert_query->execute();
      }
    }
    catch (\Exception $e) {
      if (isset($transaction)) {
        $transaction->rollBack();
      }
      Error::logException($this->logger, $e);
      throw $e;
    }

    // Clear the static caches (copied from core's implementation).
    // Note: Core uses $this->associatedRevisions and $this->associatedInitialRevisions
    // but we don't have those properties, so we'll delegate to inner for cache clearing.
    if (method_exists($this->innerAssociation, 'clearCaches')) {
      $this->innerAssociation->clearCaches();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function workspaceInsert(WorkspaceInterface $workspace) {
    return $this->innerAssociation->workspaceInsert($workspace);
  }

  /**
   * {@inheritdoc}
   */
  public function getTrackedEntities($workspace_id, $entity_type_id = NULL, $entity_ids = NULL) {
    return $this->innerAssociation->getTrackedEntities($workspace_id, $entity_type_id, $entity_ids);
  }

  /**
   * {@inheritdoc}
   */
  public function getTrackedEntitiesForListing($workspace_id, ?int $pager_id = NULL, int|false $limit = 50): array {
    return $this->innerAssociation->getTrackedEntitiesForListing($workspace_id, $pager_id, $limit);
  }

  /**
   * {@inheritdoc}
   */
  public function getAssociatedRevisions($workspace_id, $entity_type_id, $entity_ids = NULL) {
    return $this->innerAssociation->getAssociatedRevisions($workspace_id, $entity_type_id, $entity_ids);
  }

  /**
   * {@inheritdoc}
   */
  public function getAssociatedInitialRevisions(string $workspace_id, string $entity_type_id, array $entity_ids = []) {
    return $this->innerAssociation->getAssociatedInitialRevisions($workspace_id, $entity_type_id, $entity_ids);
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityTrackingWorkspaceIds(RevisionableInterface $entity, bool $latest_revision = FALSE) {
    // During revert operations, use normal tracking without parallel bypass.
    if ($this->isReverting) {
      return $this->innerAssociation->getEntityTrackingWorkspaceIds($entity, $latest_revision);
    }

    // Skip menu entities for wse_menu compatibility.
    $menu_entity_types = ['menu', 'menu_link_content'];
    if (in_array($entity->getEntityTypeId(), $menu_entity_types, TRUE)) {
      return $this->innerAssociation->getEntityTrackingWorkspaceIds($entity, $latest_revision);
    }

    // For parallel editing: bypass the workspace conflict check if this entity
    // is tracked in workspace_association for multiple workspaces OR has active
    // sessions in other workspaces. This allows parallel editing across workspaces.
    try {
      $workspace_manager = \Drupal::service('workspaces.manager');
      if ($workspace_manager->hasActiveWorkspace()) {
        $active_workspace_id = $workspace_manager->getActiveWorkspace()->id();

        // First, get which workspaces are tracking this entity.
        $tracking_workspaces = $this->innerAssociation->getEntityTrackingWorkspaceIds($entity, $latest_revision);

        // If the entity is tracked in OTHER workspaces (not just the current one),
        // allow parallel editing by returning empty array.
        if (!empty($tracking_workspaces)) {
          // Check if any tracking workspace is different from the current workspace.
          foreach ($tracking_workspaces as $tracking_workspace_id) {
            if ($tracking_workspace_id !== $active_workspace_id) {
              // Entity is tracked in another workspace - allow parallel editing.
              return [];
            }
          }
        }

        // Additionally, check for active sessions in other workspaces.
        // This covers cases where entities are being edited but not yet saved.
        $has_parallel_sessions = $this->database->select('wse_parallel_edit_session', 's')
          ->fields('s', ['sid'])
          ->condition('entity_type', $entity->getEntityTypeId())
          ->condition('entity_id', $entity->id())
          // Look for sessions in workspaces OTHER than the current one.
          ->condition('workspace_id', $active_workspace_id, '!=')
          ->range(0, 1)
          ->execute()
          ->fetchField();

        if ($has_parallel_sessions) {
          // Return empty array to allow parallel editing.
          return [];
        }
      }
    }
    catch (\Exception $e) {
      // If workspace manager or database isn't available yet (during container build),
      // fall through to default behavior.
    }

    // For all other cases, use normal workspace tracking behavior.
    return $this->innerAssociation->getEntityTrackingWorkspaceIds($entity, $latest_revision);
  }

  /**
   * {@inheritdoc}
   */
  public function postPublish(WorkspaceInterface $workspace) {
    return $this->innerAssociation->postPublish($workspace);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAssociations($workspace_id = NULL, $entity_type_id = NULL, $entity_ids = NULL, $revision_ids = NULL) {
    return $this->innerAssociation->deleteAssociations($workspace_id, $entity_type_id, $entity_ids, $revision_ids);
  }

  /**
   * {@inheritdoc}
   */
  public function initializeWorkspace(WorkspaceInterface $workspace) {
    return $this->innerAssociation->initializeWorkspace($workspace);
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Workspace association records cleanup should happen as late as possible.
    $events[WorkspacePostPublishEvent::class][] = ['onPostPublish', -500];
    return $events;
  }

  /**
   * Handles the post-publish event.
   *
   * This is a custom method added by WSE that's not part of the core
   * WorkspaceAssociationInterface. We need to support it for compatibility.
   *
   * @param \Drupal\workspaces\Event\WorkspacePostPublishEvent $event
   *   The workspace post-publish event.
   */
  public function onPostPublish(WorkspacePostPublishEvent $event): void {
    // Check if the inner service has this method (WSE adds it).
    if (method_exists($this->innerAssociation, 'onPostPublish')) {
      $this->innerAssociation->onPostPublish($event);
    }
  }

}
