<?php

namespace Drupal\wse_parallel;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\wse_parallel\ValueObject\ParallelStateDto;

/**
 * Static facade for WSE Parallel API.
 *
 * Provides convenient access to parallel editing state and operations.
 * This class is designed for use by other modules integrating with
 * WSE Parallel functionality.
 *
 * @phpstan-type SessionArray array{sid: int, entity_type: string, entity_id: int, workspace_id: string, uid: int, started: int, last_seen: int}
 * @phpstan-type PublishArray array{plid: int, entity_type: string, entity_id: int, from_revision_id: int, to_revision_id: int, workspace_id: string, published: int, langcode: string|null}
 */
class WseParallel {

  /**
   * Gets the parallel editing state for an entity.
   *
   * This method provides a comprehensive view of parallel activity including:
   * - Whether newer publishes have occurred since a given timestamp
   * - Active editing sessions from other users
   * - Latest publish information
   * - Divergence detection
   * - Suggested actions
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   * @param int|null $baseTimestamp
   *   Optional base timestamp to check for newer publishes.
   *   If NULL, uses current time.
   * @param \Drupal\Core\Session\AccountInterface|null $excludeAccount
   *   Optional account to exclude from active sessions (typically current user).
   *
   * @return \Drupal\wse_parallel\ValueObject\ParallelStateDto
   *   The parallel state data transfer object.
   *
   * @phpstan-return ParallelStateDto
   *
   * @example
   * ```php
   * use Drupal\wse_parallel\WseParallel;
   *
   * // Check if entity has parallel activity.
   * $state = WseParallel::getEntityParallelState($node);
   * if ($state->hasDiverged()) {
   *   \Drupal::messenger()->addWarning(
   *     t('This content has been modified in parallel workspaces.')
   *   );
   * }
   *
   * // Check for newer publishes since session started.
   * $sessionStart = 1234567890;
   * $state = WseParallel::getEntityParallelState($node, $sessionStart);
   * if ($state->hasNewerPublish()) {
   *   $latest = $state->getLastPublished();
   *   drupal_set_message(t('Newer version published at @time', [
   *     '@time' => date('Y-m-d H:i:s', $latest['published']),
   *   ]));
   * }
   * ```
   */
  public static function getEntityParallelState(ContentEntityInterface $entity, ?int $baseTimestamp = NULL, ?AccountInterface $excludeAccount = NULL): ParallelStateDto {
    $sessionTracker = \Drupal::service('wse_parallel.session_tracker');
    $publishLookup = \Drupal::service('wse_parallel.publish_lookup');
    $currentUser = $excludeAccount ?? \Drupal::currentUser();

    $entityType = $entity->getEntityTypeId();
    $entityId = $entity->id();
    $baseTimestamp = $baseTimestamp ?? \Drupal::time()->getRequestTime();

    // Get active sessions.
    $allSessions = $sessionTracker->getActiveSessions($entity);
    
    // Filter out the excluded account if provided.
    $activeSessions = [];
    foreach ($allSessions as $session) {
      if (!$excludeAccount || $session['uid'] != $excludeAccount->id()) {
        $activeSessions[] = $session;
      }
    }

    // Check for newer publishes.
    $hasNewerPublish = $publishLookup->hasPublishSince(
      $entityType,
      $entityId,
      $baseTimestamp
    );

    // Get latest publish.
    $lastPublished = $publishLookup->latestPublish($entityType, $entityId);

    // Determine if diverged from base.
    $divergedFromBase = FALSE;
    if ($entity->getEntityType()->isRevisionable() && $lastPublished) {
      $currentRevisionId = $entity->getRevisionId();
      $publishedRevisionId = $lastPublished['to_revision_id'];
      $divergedFromBase = ($currentRevisionId != $publishedRevisionId);
    }

    // Suggest action based on state.
    $suggestedAction = self::determineSuggestedAction(
      $hasNewerPublish,
      count($activeSessions),
      $divergedFromBase
    );

    return new ParallelStateDto(
      $hasNewerPublish,
      $activeSessions,
      $lastPublished,
      $divergedFromBase,
      $suggestedAction
    );
  }

  /**
   * Marks editing sessions as closed after a successful publish.
   *
   * This is typically called after a workspace publish operation to clean up
   * active editing sessions for the published entities.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity that was published.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account whose sessions should be closed.
   *
   * @return void
   *
   * @example
   * ```php
   * use Drupal\wse_parallel\WseParallel;
   *
   * // After publishing a workspace.
   * $publisher->publish();
   *
   * // Close the user's editing session.
   * WseParallel::markSessionClosedOnPublish($node, \Drupal::currentUser());
   * ```
   */
  public static function markSessionClosedOnPublish(ContentEntityInterface $entity, AccountInterface $account): void {
    $sessionTracker = \Drupal::service('wse_parallel.session_tracker');
    $sessionTracker->closeSessions($entity, $account);
  }

  /**
   * Determines the suggested action based on parallel state.
   *
   * @param bool $hasNewerPublish
   *   Whether a newer publish exists.
   * @param int $activeSessionCount
   *   Number of active sessions.
   * @param bool $divergedFromBase
   *   Whether the entity has diverged.
   *
   * @return string
   *   Suggested action: 'none', 'review', 'merge', 'conflict'.
   */
  private static function determineSuggestedAction(bool $hasNewerPublish, int $activeSessionCount, bool $divergedFromBase): string {
    if ($hasNewerPublish && $divergedFromBase) {
      return 'conflict';
    }

    if ($hasNewerPublish || $activeSessionCount > 0) {
      return 'review';
    }

    if ($divergedFromBase) {
      return 'merge';
    }

    return 'none';
  }

}
