<?php

namespace Drupal\Tests\wse_parallel\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;
use Drupal\workspaces\Entity\Workspace;

/**
 * Tests parallel editing UI warnings and badges.
 *
 * @group wse_parallel
 * @requires module workspaces
 */
class ParallelEditingUiTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'field',
    'text',
    'filter',
    'workspaces',
    'workspaces_ui',
    'wse_parallel',
  ];

  /**
   * A test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $testNode;

  /**
   * Workspace foo.
   *
   * @var \Drupal\workspaces\WorkspaceInterface
   */
  protected $workspaceFoo;

  /**
   * Workspace bar.
   *
   * @var \Drupal\workspaces\WorkspaceInterface
   */
  protected $workspaceBar;

  /**
   * Test user 1.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $user1;

  /**
   * Test user 2.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $user2;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create article content type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Grant permissions.
    $role = Role::load('authenticated');
    $role->grantPermission('create article content');
    $role->grantPermission('edit any article content');
    $role->grantPermission('access content');
    $role->grantPermission('view own unpublished content');
    $role->save();

    // Create workspaces.
    $this->workspaceFoo = Workspace::create([
      'id' => 'foo',
      'label' => 'Workspace Foo',
    ]);
    $this->workspaceFoo->save();

    $this->workspaceBar = Workspace::create([
      'id' => 'bar',
      'label' => 'Workspace Bar',
    ]);
    $this->workspaceBar->save();

    // Create test users.
    $this->user1 = $this->createUser([
      'create article content',
      'edit any article content',
      'access content',
    ]);

    $this->user2 = $this->createUser([
      'create article content',
      'edit any article content',
      'access content',
    ]);

    // Create a base node.
    $this->testNode = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
    ]);
    $this->testNode->save();
  }

  /**
   * Tests UI warnings when parallel editing occurs.
   */
  public function testParallelEditingWarnings() {
    // Enable divergence warnings.
    $config = \Drupal::configFactory()->getEditable('wse_parallel.settings');
    $config->set('show_publish_divergence_warning', TRUE)->save();

    // User 1 logs in and switches to workspace foo.
    $this->drupalLogin($this->user1);
    $this->switchToWorkspace('foo');

    // User 1 starts editing the node.
    $this->drupalGet('node/' . $this->testNode->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);

    // Modify and save in workspace foo.
    $page = $this->getSession()->getPage();
    $page->fillField('title[0][value]', 'Modified in Foo');
    $page->pressButton('Save');
    $this->assertSession()->pageTextContains('Article Modified in Foo has been updated.');

    // Get the workspace manager and publish foo.
    $workspace_manager = \Drupal::service('workspaces.manager');
    $workspace_manager->setActiveWorkspace($this->workspaceFoo);
    $publisher = \Drupal::service('workspaces.operation_factory')
      ->getPublisher($this->workspaceFoo);
    $publisher->publish();

    // Now simulate user 2 in workspace bar.
    $this->drupalLogout();
    $this->drupalLogin($this->user2);
    $this->switchToWorkspace('bar');

    // User 2 edits the same node (which now has a newer published version).
    $this->drupalGet('node/' . $this->testNode->id() . '/edit');

    // Wait for the page to load.
    $this->assertSession()->waitForElement('css', 'form.node-article-edit-form');

    // Check for warning message about newer publish.
    $this->assertSession()->pageTextContains('Another workspace');
    $this->assertSession()->pageTextContains('published a newer revision');

    // Check for the parallel activity badge.
    $this->assertSession()->elementExists('css', '.wse-parallel-badge');
  }

  /**
   * Tests session tracking across multiple users.
   */
  public function testMultipleUserSessions() {
    // User 1 starts editing in workspace foo.
    $this->drupalLogin($this->user1);
    $this->switchToWorkspace('foo');
    $this->drupalGet('node/' . $this->testNode->id() . '/edit');

    // Verify session is created.
    $sessions = \Drupal::database()->select('wse_parallel_edit_session', 's')
      ->fields('s')
      ->condition('entity_type', 'node')
      ->condition('entity_id', $this->testNode->id())
      ->condition('uid', $this->user1->id())
      ->condition('status', 'active')
      ->execute()
      ->fetchAll();

    $this->assertNotEmpty($sessions, 'Session should be created for user 1.');

    // User 2 starts editing in workspace bar.
    $this->drupalLogout();
    $this->drupalLogin($this->user2);
    $this->switchToWorkspace('bar');
    $this->drupalGet('node/' . $this->testNode->id() . '/edit');

    // Verify second session is created.
    $sessions = \Drupal::database()->select('wse_parallel_edit_session', 's')
      ->fields('s')
      ->condition('entity_type', 'node')
      ->condition('entity_id', $this->testNode->id())
      ->condition('status', 'active')
      ->execute()
      ->fetchAll();

    $this->assertCount(2, $sessions, 'Two active sessions should exist.');
  }

  /**
   * Tests the parallel activity report page.
   */
  public function testParallelActivityReport() {
    // Create some activity.
    $this->drupalLogin($this->user1);
    $this->switchToWorkspace('foo');
    $this->drupalGet('node/' . $this->testNode->id() . '/edit');

    // Visit the activity report.
    $this->drupalGet('wse-parallel/activity/node/' . $this->testNode->id());
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Parallel Activity Report');
    $this->assertSession()->pageTextContains('Active Editing Sessions');
  }

  /**
   * Helper to switch to a workspace.
   *
   * @param string $workspace_id
   *   The workspace ID.
   */
  protected function switchToWorkspace(string $workspace_id): void {
    $workspace = Workspace::load($workspace_id);
    if ($workspace) {
      $workspace_manager = \Drupal::service('workspaces.manager');
      $workspace_manager->setActiveWorkspace($workspace);
    }
  }

}
