<?php

namespace Drupal\Tests\wse_parallel\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\workspaces\Entity\Workspace;

/**
 * Tests conflict resolution strategies.
 *
 * @group wse_parallel
 */
class ConflictResolutionTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'workspaces',
    'wse_parallel',
  ];

  /**
   * The conflict strategy manager.
   *
   * @var \Drupal\wse_parallel\Conflict\ConflictStrategyManager
   */
  protected $strategyManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('workspace');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('system', ['sequences']);
    $this->installConfig(['filter', 'node', 'system', 'wse_parallel']);

    $this->strategyManager = $this->container->get('wse_parallel.conflict_strategy_manager');

    // Create a node type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();
  }

  /**
   * Tests that the default strategy is available.
   */
  public function testDefaultStrategyAvailable() {
    $definitions = $this->strategyManager->getDefinitions();
    $this->assertArrayHasKey('last_publish_wins', $definitions);

    $strategy = $this->strategyManager->createInstance('last_publish_wins');
    $this->assertInstanceOf(
      'Drupal\wse_parallel\Conflict\ConflictStrategyInterface',
      $strategy
    );
  }

  /**
   * Tests getting the active strategy.
   */
  public function testGetActiveStrategy() {
    $strategy = $this->strategyManager->getActiveStrategy();
    $this->assertInstanceOf(
      'Drupal\wse_parallel\Conflict\ConflictStrategyInterface',
      $strategy
    );
    $this->assertEquals('Last Publish Wins', $strategy->label());
  }

  /**
   * Tests strategy options for form select.
   */
  public function testGetStrategyOptions() {
    $options = $this->strategyManager->getStrategyOptions();
    $this->assertIsArray($options);
    $this->assertArrayHasKey('last_publish_wins', $options);
    $this->assertEquals('Last Publish Wins', $options['last_publish_wins']);
  }

  /**
   * Tests last publish wins strategy behavior.
   */
  public function testLastPublishWinsStrategy() {
    $strategy = $this->strategyManager->createInstance('last_publish_wins');

    // Create a workspace and node.
    $workspace = Workspace::create([
      'id' => 'test',
      'label' => 'Test Workspace',
    ]);
    $workspace->save();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Node',
      'status' => 1,
    ]);
    $node->save();

    // Test pre-publish.
    $context = [
      'workspace' => $workspace,
      'published_revision_ids' => [
        'node' => [100 => $node->id()],
      ],
    ];

    $result = $strategy->onPrePublish($context);
    $this->assertTrue($result->isSuccess());
    $this->assertEquals('validated', $result->getAction());

    // Test post-publish.
    $result = $strategy->onPostPublish($context);
    $this->assertTrue($result->isSuccess());
    $this->assertEquals('published', $result->getAction());

    // Test resolve.
    $resolve_context = [
      'workspace' => $workspace,
      'workspace_revision_id' => 100,
      'base_revision_id' => 50,
    ];

    $result = $strategy->resolve($node, $resolve_context);
    $this->assertTrue($result->isSuccess());
    $this->assertEquals('overwrite', $result->getAction());
  }

}
