<?php

namespace Drupal\Tests\wse_parallel\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;
use Drupal\workspaces\Entity\Workspace;

/**
 * Tests parallel entity access control.
 *
 * @group wse_parallel
 */
class ParallelEntityAccessTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'workspaces',
    'wse_parallel',
  ];

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The workspace manager.
   *
   * @var \Drupal\workspaces\WorkspaceManagerInterface
   */
  protected $workspaceManager;

  /**
   * The parallel entity access service.
   *
   * @var \Drupal\wse_parallel\Access\ParallelEntityAccess
   */
  protected $accessChecker;

  /**
   * Test user with bypass permission.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $bypassUser;

  /**
   * Test user without bypass permission.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $normalUser;

  /**
   * Test workspace.
   *
   * @var \Drupal\workspaces\WorkspaceInterface
   */
  protected $workspace;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('workspace');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('system', ['sequences']);
    $this->installConfig(['filter', 'node', 'system']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->workspaceManager = $this->container->get('workspaces.manager');
    $this->accessChecker = $this->container->get('wse_parallel.parallel_entity_access');

    // Create a node type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Create a role with bypass permission.
    $bypass_role = Role::create([
      'id' => 'bypass_role',
      'label' => 'Bypass Role',
    ]);
    $bypass_role->grantPermission('bypass wse parallel guards');
    $bypass_role->save();

    // Create users.
    $this->bypassUser = User::create([
      'name' => 'bypass_user',
      'mail' => 'bypass@example.com',
      'status' => 1,
    ]);
    $this->bypassUser->addRole('bypass_role');
    $this->bypassUser->save();

    $this->normalUser = User::create([
      'name' => 'normal_user',
      'mail' => 'normal@example.com',
      'status' => 1,
    ]);
    $this->normalUser->save();

    // Create a workspace.
    $this->workspace = Workspace::create([
      'id' => 'stage',
      'label' => 'Stage',
    ]);
    $this->workspace->save();

    // Activate the workspace.
    $this->workspaceManager->setActiveWorkspace($this->workspace);

    // Create a test node.
    $this->node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
    ]);
    $this->node->save();
  }

  /**
   * Tests that bypass permission grants access.
   */
  public function testBypassPermissionGrantsAccess() {
    $result = $this->accessChecker->checkAccess(
      $this->node,
      'revert',
      $this->bypassUser
    );

    $this->assertTrue($result->isAllowed(), 'User with bypass permission should have access to revert.');
  }

  /**
   * Tests that normal users get neutral result.
   */
  public function testNormalUserGetsNeutral() {
    $result = $this->accessChecker->checkAccess(
      $this->node,
      'revert',
      $this->normalUser
    );

    $this->assertTrue($result->isNeutral(), 'User without bypass permission should get neutral result.');
  }

  /**
   * Tests that non-restricted operations return neutral.
   */
  public function testNonRestrictedOperationsReturnNeutral() {
    $result = $this->accessChecker->checkAccess(
      $this->node,
      'view',
      $this->bypassUser
    );

    $this->assertTrue($result->isNeutral(), 'Non-restricted operations should return neutral.');
  }

  /**
   * Tests all restricted operations.
   */
  public function testAllRestrictedOperations() {
    $operations = ['revert', 'revert revision', 'delete revision'];

    foreach ($operations as $operation) {
      $result = $this->accessChecker->checkAccess(
        $this->node,
        $operation,
        $this->bypassUser
      );

      $this->assertTrue(
        $result->isAllowed(),
        "User with bypass permission should have access to {$operation}."
      );
    }
  }

  /**
   * Tests workspace lineage access.
   */
  public function testWorkspaceLineageAccess() {
    // Create a parent workspace.
    $parent_workspace = Workspace::create([
      'id' => 'parent',
      'label' => 'Parent',
    ]);
    $parent_workspace->save();

    // Create a child workspace.
    $child_workspace = Workspace::create([
      'id' => 'child',
      'label' => 'Child',
      'parent' => 'parent',
    ]);
    $child_workspace->save();

    // Set the child workspace as active.
    $this->workspaceManager->setActiveWorkspace($child_workspace);

    // Create a node in the parent workspace context.
    $this->workspaceManager->setActiveWorkspace($parent_workspace);
    $node_in_parent = Node::create([
      'type' => 'article',
      'title' => 'Node in Parent',
      'status' => 1,
    ]);
    $node_in_parent->save();

    // Switch back to child workspace.
    $this->workspaceManager->setActiveWorkspace($child_workspace);

    // Test access from child workspace to parent workspace entity.
    $result = $this->accessChecker->checkAccess(
      $node_in_parent,
      'revert',
      $this->normalUser
    );

    // This should be allowed because child is in the lineage of parent.
    $this->assertTrue(
      $result->isAllowed() || $result->isNeutral(),
      'Access from child workspace to parent workspace entity should be allowed or neutral.'
    );
  }

  /**
   * Tests that non-revisionable entities return neutral.
   */
  public function testNonRevisionableEntityReturnsNeutral() {
    // Users are not revisionable.
    $result = $this->accessChecker->checkAccess(
      $this->normalUser,
      'revert',
      $this->bypassUser
    );

    $this->assertTrue($result->isNeutral(), 'Non-revisionable entities should return neutral.');
  }

}
