<?php

namespace Drupal\Tests\wse_parallel\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;
use Drupal\workspaces\Entity\Workspace;
use Drupal\wse_parallel\WseParallel;

/**
 * Tests parallel editing workflows.
 *
 * @group wse_parallel
 * @requires module workspaces
 */
class ParallelWorkflowTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'workspaces',
    'workspaces_ui',
    'wse_parallel',
  ];

  /**
   * The workspace manager.
   *
   * @var \Drupal\workspaces\WorkspaceManagerInterface
   */
  protected $workspaceManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The publish lookup service.
   *
   * @var \Drupal\wse_parallel\Publish\PublishLookup
   */
  protected $publishLookup;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Test user with bypass permission.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $bypassUser;

  /**
   * Test user without bypass permission.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $normalUser;

  /**
   * Base node for testing.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $baseNode;

  /**
   * Workspace foo.
   *
   * @var \Drupal\workspaces\WorkspaceInterface
   */
  protected $workspaceFoo;

  /**
   * Workspace bar.
   *
   * @var \Drupal\workspaces\WorkspaceInterface
   */
  protected $workspaceBar;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('workspace');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('system', ['sequences']);
    $this->installSchema('wse_parallel', [
      'wse_parallel_edit_session',
      'wse_parallel_publish_log',
    ]);
    $this->installConfig(['filter', 'node', 'system', 'workspaces', 'wse_parallel']);

    $this->workspaceManager = $this->container->get('workspaces.manager');
    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->publishLookup = $this->container->get('wse_parallel.publish_lookup');
    $this->database = $this->container->get('database');

    // Create a node type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Create a role with bypass permission.
    $bypass_role = Role::create([
      'id' => 'bypass_role',
      'label' => 'Bypass Role',
    ]);
    $bypass_role->grantPermission('bypass wse parallel guards');
    $bypass_role->grantPermission('view own unpublished content');
    $bypass_role->grantPermission('edit any article content');
    $bypass_role->save();

    // Create users.
    $this->bypassUser = User::create([
      'name' => 'bypass_user',
      'mail' => 'bypass@example.com',
      'status' => 1,
    ]);
    $this->bypassUser->addRole('bypass_role');
    $this->bypassUser->save();

    $this->normalUser = User::create([
      'name' => 'normal_user',
      'mail' => 'normal@example.com',
      'status' => 1,
    ]);
    $this->normalUser->save();

    // Create base node in default workspace.
    $this->workspaceManager->setActiveWorkspace(NULL);
    $this->baseNode = Node::create([
      'type' => 'article',
      'title' => 'Base Article',
      'status' => 1,
    ]);
    $this->baseNode->save();

    // Create two workspaces.
    $this->workspaceFoo = Workspace::create([
      'id' => 'foo',
      'label' => 'Workspace Foo',
    ]);
    $this->workspaceFoo->save();

    $this->workspaceBar = Workspace::create([
      'id' => 'bar',
      'label' => 'Workspace Bar',
    ]);
    $this->workspaceBar->save();
  }

  /**
   * Tests simultaneous edits in two workspaces without conflict errors.
   */
  public function testSimultaneousEditsNoConflict() {
    $base_revision_id = $this->baseNode->getRevisionId();

    // Edit in workspace foo.
    $this->workspaceManager->setActiveWorkspace($this->workspaceFoo);
    $node_foo = $this->entityTypeManager->getStorage('node')->load($this->baseNode->id());
    $node_foo->setTitle('Modified in Foo');
    $node_foo->setNewRevision(TRUE);
    $node_foo->save();
    $foo_revision_id = $node_foo->getRevisionId();

    // Edit in workspace bar (based on same base revision).
    $this->workspaceManager->setActiveWorkspace($this->workspaceBar);
    $node_bar = $this->entityTypeManager->getStorage('node')->load($this->baseNode->id());
    $node_bar->setTitle('Modified in Bar');
    $node_bar->setNewRevision(TRUE);
    $node_bar->save();
    $bar_revision_id = $node_bar->getRevisionId();

    // Both saves should succeed without validation errors.
    $this->assertNotEquals($base_revision_id, $foo_revision_id);
    $this->assertNotEquals($base_revision_id, $bar_revision_id);
    $this->assertNotEquals($foo_revision_id, $bar_revision_id);

    // Verify both revisions exist.
    $storage = $this->entityTypeManager->getStorage('node');
    $foo_loaded = $storage->loadRevision($foo_revision_id);
    $bar_loaded = $storage->loadRevision($bar_revision_id);

    $this->assertEquals('Modified in Foo', $foo_loaded->getTitle());
    $this->assertEquals('Modified in Bar', $bar_loaded->getTitle());
  }

  /**
   * Tests publish logging and divergence warnings.
   */
  public function testPublishLoggingAndDivergenceWarnings() {
    $base_revision_id = $this->baseNode->getRevisionId();

    // Edit in workspace foo.
    $this->workspaceManager->setActiveWorkspace($this->workspaceFoo);
    $node_foo = $this->entityTypeManager->getStorage('node')->load($this->baseNode->id());
    $node_foo->setTitle('Modified in Foo');
    $node_foo->setNewRevision(TRUE);
    $node_foo->save();
    $foo_revision_id = $node_foo->getRevisionId();

    // Edit in workspace bar.
    $this->workspaceManager->setActiveWorkspace($this->workspaceBar);
    $node_bar = $this->entityTypeManager->getStorage('node')->load($this->baseNode->id());
    $node_bar->setTitle('Modified in Bar');
    $node_bar->setNewRevision(TRUE);
    $node_bar->save();
    $bar_revision_id = $node_bar->getRevisionId();

    // Publish workspace foo.
    $publisher_foo = $this->container->get('workspaces.operation_factory')
      ->getPublisher($this->workspaceFoo);
    $publisher_foo->publish();

    // Verify publish log entry exists.
    $publish_records = $this->database->select('wse_parallel_publish_log', 'p')
      ->fields('p')
      ->condition('entity_type', 'node')
      ->condition('entity_id', $this->baseNode->id())
      ->condition('workspace_id', 'foo')
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);

    $this->assertCount(1, $publish_records, 'Publish log entry should exist for foo.');
    $record = reset($publish_records);
    $this->assertEquals($base_revision_id, $record['from_revision_id']);
    $this->assertEquals($foo_revision_id, $record['to_revision_id']);

    // Check parallel state in workspace bar (should show divergence).
    $this->workspaceManager->setActiveWorkspace($this->workspaceBar);
    $state = WseParallel::getEntityParallelState($node_bar, time() - 3600);

    $this->assertTrue($state->hasNewerPublish(), 'Should detect newer publish from foo.');
    $this->assertTrue($state->hasDiverged(), 'Should detect divergence.');
    $this->assertEquals('conflict', $state->getSuggestedAction());
  }

  /**
   * Tests last publish wins strategy.
   */
  public function testLastPublishWinsStrategy() {
    // Edit in workspace foo.
    $this->workspaceManager->setActiveWorkspace($this->workspaceFoo);
    $node_foo = $this->entityTypeManager->getStorage('node')->load($this->baseNode->id());
    $node_foo->setTitle('Modified in Foo');
    $node_foo->setNewRevision(TRUE);
    $node_foo->save();
    $foo_revision_id = $node_foo->getRevisionId();

    // Edit in workspace bar.
    $this->workspaceManager->setActiveWorkspace($this->workspaceBar);
    $node_bar = $this->entityTypeManager->getStorage('node')->load($this->baseNode->id());
    $node_bar->setTitle('Modified in Bar');
    $node_bar->setNewRevision(TRUE);
    $node_bar->save();
    $bar_revision_id = $node_bar->getRevisionId();

    // Publish foo first.
    $publisher_foo = $this->container->get('workspaces.operation_factory')
      ->getPublisher($this->workspaceFoo);
    $publisher_foo->publish();

    // Verify default revision is now foo's revision.
    $this->workspaceManager->setActiveWorkspace(NULL);
    $default_node = $this->entityTypeManager->getStorage('node')->load($this->baseNode->id());
    $this->assertEquals('Modified in Foo', $default_node->getTitle());
    $this->assertEquals($foo_revision_id, $default_node->getRevisionId());

    // Publish bar (should overwrite foo's changes).
    $publisher_bar = $this->container->get('workspaces.operation_factory')
      ->getPublisher($this->workspaceBar);
    $publisher_bar->publish();

    // Verify default revision is now bar's revision (last publish wins).
    $default_node = $this->entityTypeManager->getStorage('node')->load($this->baseNode->id());
    $this->assertEquals('Modified in Bar', $default_node->getTitle());
    $this->assertEquals($bar_revision_id, $default_node->getRevisionId());

    // Verify both publish log entries exist.
    $total_records = $this->database->select('wse_parallel_publish_log', 'p')
      ->condition('entity_type', 'node')
      ->condition('entity_id', $this->baseNode->id())
      ->countQuery()
      ->execute()
      ->fetchField();

    $this->assertEquals(2, $total_records, 'Should have two publish log entries.');
  }

  /**
   * Tests revert/delete revision access control.
   */
  public function testRevertDeleteRevisionAccess() {
    // Create a revision in workspace foo.
    $this->workspaceManager->setActiveWorkspace($this->workspaceFoo);
    $node = $this->entityTypeManager->getStorage('node')->load($this->baseNode->id());
    $node->setTitle('Modified in Foo');
    $node->setNewRevision(TRUE);
    $node->save();

    // Test with bypass user.
    $access_checker = $this->container->get('wse_parallel.parallel_entity_access');
    
    $revert_access = $access_checker->checkAccess($node, 'revert', $this->bypassUser);
    $this->assertTrue(
      $revert_access->isAllowed(),
      'User with bypass permission should have revert access.'
    );

    $delete_revision_access = $access_checker->checkAccess(
      $node,
      'delete revision',
      $this->bypassUser
    );
    $this->assertTrue(
      $delete_revision_access->isAllowed(),
      'User with bypass permission should have delete revision access.'
    );

    // Test with normal user (should get neutral).
    $normal_revert_access = $access_checker->checkAccess($node, 'revert', $this->normalUser);
    $this->assertTrue(
      $normal_revert_access->isNeutral(),
      'Normal user should get neutral access result.'
    );
  }

  /**
   * Tests workspace lineage access.
   */
  public function testWorkspaceLineageAccess() {
    // Create parent workspace.
    $parent = Workspace::create([
      'id' => 'parent',
      'label' => 'Parent Workspace',
    ]);
    $parent->save();

    // Create child workspace.
    $child = Workspace::create([
      'id' => 'child',
      'label' => 'Child Workspace',
      'parent' => 'parent',
    ]);
    $child->save();

    // Create node in parent workspace.
    $this->workspaceManager->setActiveWorkspace($parent);
    $node = Node::create([
      'type' => 'article',
      'title' => 'Parent Node',
      'status' => 1,
    ]);
    $node->save();

    // Switch to child workspace.
    $this->workspaceManager->setActiveWorkspace($child);

    // Access from child to parent's entity should be allowed.
    $access_checker = $this->container->get('wse_parallel.parallel_entity_access');
    $access = $access_checker->checkAccess($node, 'revert', $this->normalUser);

    // Should be allowed or neutral (lineage check).
    $this->assertTrue(
      $access->isAllowed() || $access->isNeutral(),
      'Access from child workspace should be allowed or neutral.'
    );
  }

}
