<?php

namespace Drupal\Tests\wse_parallel\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\workspaces\Entity\Workspace;

/**
 * Tests the publish logger service.
 *
 * @group wse_parallel
 */
class PublishLoggerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'workspaces',
    'wse_parallel',
  ];

  /**
   * The workspace manager.
   *
   * @var \Drupal\workspaces\WorkspaceManagerInterface
   */
  protected $workspaceManager;

  /**
   * The workspace publisher.
   *
   * @var \Drupal\workspaces\WorkspacePublisherInterface
   */
  protected $workspacePublisher;

  /**
   * The publish lookup service.
   *
   * @var \Drupal\wse_parallel\Publish\PublishLookup
   */
  protected $publishLookup;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('workspace');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('system', ['sequences']);
    $this->installSchema('wse_parallel', [
      'wse_parallel_edit_session',
      'wse_parallel_publish_log',
    ]);
    $this->installConfig(['filter', 'node', 'system', 'workspaces']);

    $this->workspaceManager = $this->container->get('workspaces.manager');
    $this->workspacePublisher = $this->container->get('workspaces.operation_factory')->getPublisher($this->workspaceManager->getActiveWorkspace());
    $this->publishLookup = $this->container->get('wse_parallel.publish_lookup');
    $this->database = $this->container->get('database');

    // Create a node type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();
  }

  /**
   * Tests basic publish logging.
   */
  public function testBasicPublishLogging() {
    // Create a base node in the default workspace.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Base Article',
      'status' => 1,
    ]);
    $node->save();
    $base_revision_id = $node->getRevisionId();

    // Create a workspace and make changes.
    $workspace_foo = Workspace::create([
      'id' => 'foo',
      'label' => 'Foo Workspace',
    ]);
    $workspace_foo->save();

    // Switch to the workspace and edit the node.
    $this->workspaceManager->setActiveWorkspace($workspace_foo);
    $node->setTitle('Modified in Foo');
    $node->setNewRevision(TRUE);
    $node->save();
    $foo_revision_id = $node->getRevisionId();

    // Publish the workspace.
    $publisher = $this->container->get('workspaces.operation_factory')->getPublisher($workspace_foo);
    $publisher->publish();

    // Check that a publish log record was created.
    $records = $this->database->select('wse_parallel_publish_log', 'p')
      ->fields('p')
      ->condition('entity_type', 'node')
      ->condition('entity_id', $node->id())
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);

    $this->assertCount(1, $records, 'One publish log record should exist.');
    
    $record = reset($records);
    $this->assertEquals('node', $record['entity_type']);
    $this->assertEquals($node->id(), $record['entity_id']);
    $this->assertEquals($base_revision_id, $record['from_revision_id']);
    $this->assertEquals($foo_revision_id, $record['to_revision_id']);
    $this->assertEquals('foo', $record['workspace_id']);
  }

  /**
   * Tests multiple parallel publishes.
   */
  public function testMultipleParallelPublishes() {
    // Create a base node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Base Article',
      'status' => 1,
    ]);
    $node->save();
    $base_revision_id = $node->getRevisionId();

    // Create workspace foo and edit.
    $workspace_foo = Workspace::create([
      'id' => 'foo',
      'label' => 'Foo Workspace',
    ]);
    $workspace_foo->save();

    $this->workspaceManager->setActiveWorkspace($workspace_foo);
    $node->setTitle('Modified in Foo');
    $node->setNewRevision(TRUE);
    $node->save();
    $foo_revision_id = $node->getRevisionId();

    // Switch back to default and create workspace bar.
    $this->workspaceManager->setActiveWorkspace(NULL);
    
    $workspace_bar = Workspace::create([
      'id' => 'bar',
      'label' => 'Bar Workspace',
    ]);
    $workspace_bar->save();

    $this->workspaceManager->setActiveWorkspace($workspace_bar);
    $node->setTitle('Modified in Bar');
    $node->setNewRevision(TRUE);
    $node->save();
    $bar_revision_id = $node->getRevisionId();

    // Publish foo first.
    $publisher_foo = $this->container->get('workspaces.operation_factory')->getPublisher($workspace_foo);
    $publisher_foo->publish();

    // Check first publish.
    $records = $this->database->select('wse_parallel_publish_log', 'p')
      ->fields('p')
      ->condition('entity_type', 'node')
      ->condition('entity_id', $node->id())
      ->condition('workspace_id', 'foo')
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);

    $this->assertCount(1, $records, 'One record for foo workspace.');
    $record_foo = reset($records);
    $this->assertEquals($base_revision_id, $record_foo['from_revision_id']);
    $this->assertEquals($foo_revision_id, $record_foo['to_revision_id']);

    // Now publish bar (which will overwrite foo's changes).
    $publisher_bar = $this->container->get('workspaces.operation_factory')->getPublisher($workspace_bar);
    $publisher_bar->publish();

    // Check second publish.
    $records_bar = $this->database->select('wse_parallel_publish_log', 'p')
      ->fields('p')
      ->condition('entity_type', 'node')
      ->condition('entity_id', $node->id())
      ->condition('workspace_id', 'bar')
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);

    $this->assertCount(1, $records_bar, 'One record for bar workspace.');
    $record_bar = reset($records_bar);
    
    // Bar's from_revision should be foo's to_revision (since foo published first).
    $this->assertEquals($foo_revision_id, $record_bar['from_revision_id']);
    $this->assertEquals($bar_revision_id, $record_bar['to_revision_id']);

    // Total records should be 2.
    $total_records = $this->database->select('wse_parallel_publish_log', 'p')
      ->condition('entity_type', 'node')
      ->condition('entity_id', $node->id())
      ->countQuery()
      ->execute()
      ->fetchField();

    $this->assertEquals(2, $total_records, 'Two total publish records should exist.');
  }

  /**
   * Tests PublishLookup service methods.
   */
  public function testPublishLookup() {
    // Create and publish a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
    ]);
    $node->save();

    $workspace = Workspace::create([
      'id' => 'test',
      'label' => 'Test Workspace',
    ]);
    $workspace->save();

    $this->workspaceManager->setActiveWorkspace($workspace);
    $node->setTitle('Modified');
    $node->setNewRevision(TRUE);
    $node->save();

    $publisher = $this->container->get('workspaces.operation_factory')->getPublisher($workspace);
    $publisher->publish();

    // Test hasPublishSince.
    $past_timestamp = time() - 3600;
    $this->assertTrue(
      $this->publishLookup->hasPublishSince('node', $node->id(), $past_timestamp),
      'Should find publish after past timestamp.'
    );

    $future_timestamp = time() + 3600;
    $this->assertFalse(
      $this->publishLookup->hasPublishSince('node', $node->id(), $future_timestamp),
      'Should not find publish after future timestamp.'
    );

    // Test latestPublish.
    $latest = $this->publishLookup->latestPublish('node', $node->id());
    $this->assertNotNull($latest, 'Latest publish record should exist.');
    $this->assertEquals('node', $latest['entity_type']);
    $this->assertEquals($node->id(), $latest['entity_id']);
    $this->assertEquals('test', $latest['workspace_id']);

    // Test getPublishHistory.
    $history = $this->publishLookup->getPublishHistory('node', $node->id());
    $this->assertCount(1, $history, 'Should have one publish history record.');
  }

}
