<?php

namespace Drupal\Tests\wse_parallel\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\User;
use Drupal\workspaces\Entity\Workspace;

/**
 * Tests session tracker integration.
 *
 * @group wse_parallel
 */
class SessionTrackerIntegrationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'workspaces',
    'wse_parallel',
  ];

  /**
   * The session tracker service.
   *
   * @var \Drupal\wse_parallel\Parallel\SessionTrackerInterface
   */
  protected $sessionTracker;

  /**
   * The workspace manager.
   *
   * @var \Drupal\workspaces\WorkspaceManagerInterface
   */
  protected $workspaceManager;

  /**
   * Test workspace.
   *
   * @var \Drupal\workspaces\WorkspaceInterface
   */
  protected $workspace;

  /**
   * Test user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $user;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('workspace');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('system', ['sequences']);
    $this->installSchema('wse_parallel', ['wse_parallel_edit_session', 'wse_parallel_publish_log']);
    $this->installConfig(['filter', 'node', 'system']);

    $this->sessionTracker = $this->container->get('wse_parallel.session_tracker');
    $this->workspaceManager = $this->container->get('workspaces.manager');

    // Create a node type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Create a user.
    $this->user = User::create([
      'name' => 'test_user',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $this->user->save();

    // Create a workspace.
    $this->workspace = Workspace::create([
      'id' => 'stage',
      'label' => 'Stage',
    ]);
    $this->workspace->save();

    // Activate the workspace.
    $this->workspaceManager->setActiveWorkspace($this->workspace);

    // Create a test node.
    $this->node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
    ]);
    $this->node->save();
  }

  /**
   * Tests starting a new session.
   */
  public function testStartSession() {
    $sid = $this->sessionTracker->startOrUpdateSession(
      $this->node,
      $this->node->getRevisionId(),
      $this->node->getRevisionId(),
      $this->workspace,
      $this->user
    );

    $this->assertGreaterThan(0, $sid, 'Session ID should be greater than 0.');

    // Verify session was created in database.
    $sessions = $this->sessionTracker->getActiveSessions($this->node);
    $this->assertCount(1, $sessions, 'There should be one active session.');
    $this->assertEquals($sid, reset($sessions)['sid'], 'Session ID should match.');
  }

  /**
   * Tests updating an existing session.
   */
  public function testUpdateSession() {
    // Start initial session.
    $sid1 = $this->sessionTracker->startOrUpdateSession(
      $this->node,
      1,
      1,
      $this->workspace,
      $this->user
    );

    // Update the session.
    $sid2 = $this->sessionTracker->startOrUpdateSession(
      $this->node,
      1,
      2,
      $this->workspace,
      $this->user
    );

    $this->assertEquals($sid1, $sid2, 'Session ID should be the same for updates.');

    // Verify only one active session exists.
    $sessions = $this->sessionTracker->getActiveSessions($this->node);
    $this->assertCount(1, $sessions, 'There should still be only one active session.');
  }

  /**
   * Tests closing sessions.
   */
  public function testCloseSessions() {
    // Start a session.
    $this->sessionTracker->startOrUpdateSession(
      $this->node,
      1,
      1,
      $this->workspace,
      $this->user
    );

    // Verify session is active.
    $sessions = $this->sessionTracker->getActiveSessions($this->node);
    $this->assertCount(1, $sessions, 'There should be one active session.');

    // Close the session.
    $this->sessionTracker->closeSessions($this->node, $this->user);

    // Verify session is no longer active.
    $sessions = $this->sessionTracker->getActiveSessions($this->node);
    $this->assertCount(0, $sessions, 'There should be no active sessions after closing.');
  }

  /**
   * Tests getting active sessions for multiple users.
   */
  public function testMultipleUserSessions() {
    // Create another user.
    $user2 = User::create([
      'name' => 'test_user2',
      'mail' => 'test2@example.com',
      'status' => 1,
    ]);
    $user2->save();

    // Start sessions for both users.
    $this->sessionTracker->startOrUpdateSession(
      $this->node,
      1,
      1,
      $this->workspace,
      $this->user
    );

    $this->sessionTracker->startOrUpdateSession(
      $this->node,
      1,
      1,
      $this->workspace,
      $user2
    );

    // Verify both sessions are active.
    $sessions = $this->sessionTracker->getActiveSessions($this->node);
    $this->assertCount(2, $sessions, 'There should be two active sessions.');

    // Close one user's session.
    $this->sessionTracker->closeSessions($this->node, $this->user);

    // Verify only one session remains.
    $sessions = $this->sessionTracker->getActiveSessions($this->node);
    $this->assertCount(1, $sessions, 'There should be one active session after closing one.');
  }

}
