<?php

namespace Drupal\Tests\wse_parallel\Unit\ConflictStrategy;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\wse_parallel\Plugin\WseParallelConflictStrategy\LastPublishWins;
use Drupal\workspaces\WorkspaceInterface;

/**
 * Tests the LastPublishWins conflict strategy.
 *
 * @coversDefaultClass \Drupal\wse_parallel\Plugin\WseParallelConflictStrategy\LastPublishWins
 * @group wse_parallel
 */
class LastPublishWinsTest extends UnitTestCase {

  /**
   * The strategy under test.
   *
   * @var \Drupal\wse_parallel\Plugin\WseParallelConflictStrategy\LastPublishWins
   */
  protected $strategy;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->strategy = new LastPublishWins([], 'last_publish_wins', [
      'label' => 'Last Publish Wins',
      'description' => 'Test description',
    ]);
  }

  /**
   * Tests the onPrePublish method.
   *
   * @covers ::onPrePublish
   */
  public function testOnPrePublish() {
    $workspace = $this->createMock(WorkspaceInterface::class);
    $context = [
      'workspace' => $workspace,
      'published_revision_ids' => [
        'node' => [1 => 100, 2 => 101],
      ],
    ];

    $result = $this->strategy->onPrePublish($context);

    $this->assertTrue($result->isSuccess());
    $this->assertEquals('validated', $result->getAction());
  }

  /**
   * Tests the onPostPublish method.
   *
   * @covers ::onPostPublish
   */
  public function testOnPostPublish() {
    $workspace = $this->createMock(WorkspaceInterface::class);
    $context = [
      'workspace' => $workspace,
      'published_revision_ids' => [
        'node' => [1 => 100, 2 => 101],
        'taxonomy_term' => [3 => 200],
      ],
    ];

    $result = $this->strategy->onPostPublish($context);

    $this->assertTrue($result->isSuccess());
    $this->assertEquals('published', $result->getAction());
    $this->assertEquals(3, $result->getMetadataValue('entity_count'));
  }

  /**
   * Tests the resolve method.
   *
   * @covers ::resolve
   */
  public function testResolve() {
    $entity = $this->createMock(ContentEntityInterface::class);
    $entity->method('id')->willReturn(123);
    $entity->method('getEntityTypeId')->willReturn('node');

    $context = [
      'workspace_revision_id' => 456,
      'base_revision_id' => 400,
    ];

    $result = $this->strategy->resolve($entity, $context);

    $this->assertTrue($result->isSuccess());
    $this->assertEquals('overwrite', $result->getAction());
    $this->assertEquals(123, $result->getMetadataValue('entity_id'));
    $this->assertEquals('node', $result->getMetadataValue('entity_type'));
    $this->assertEquals(456, $result->getMetadataValue('workspace_revision_id'));
  }

  /**
   * Tests the label method.
   *
   * @covers ::label
   */
  public function testLabel() {
    $label = $this->strategy->label();
    $this->assertIsString($label);
    $this->assertNotEmpty($label);
  }

  /**
   * Tests the description method.
   *
   * @covers ::description
   */
  public function testDescription() {
    $description = $this->strategy->description();
    $this->assertIsString($description);
    $this->assertNotEmpty($description);
  }

}
