<?php

namespace Drupal\Tests\wse_parallel\Unit;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Controller\EntityController;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\wse_parallel\Controller\ParallelEntityFormController;

/**
 * Tests the ParallelEntityFormController decorator.
 *
 * @coversDefaultClass \Drupal\wse_parallel\Controller\ParallelEntityFormController
 * @group wse_parallel
 */
class ParallelEntityFormControllerTest extends UnitTestCase {

  /**
   * The parallel entity form controller under test.
   *
   * @var \Drupal\wse_parallel\Controller\ParallelEntityFormController
   */
  protected $controller;

  /**
   * The mocked inner controller.
   *
   * @var \Drupal\Core\Entity\Controller\EntityController|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $innerController;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a mock inner controller.
    $this->innerController = $this->createMock(EntityController::class);

    // Create the controller under test.
    $entity_type_manager = $this->createMock('Drupal\Core\Entity\EntityTypeManagerInterface');
    $form_builder = $this->createMock('Drupal\Core\Form\FormBuilderInterface');
    
    $this->controller = new ParallelEntityFormController(
      $entity_type_manager,
      $form_builder
    );
    $this->controller->setInnerController($this->innerController);
  }

  /**
   * Tests that the decorator delegates to the inner controller.
   *
   * @covers ::editForm
   */
  public function testEditFormDelegation() {
    $entity = $this->createMockEntity('node', TRUE, FALSE);
    $expected_form = ['#form' => 'test'];

    $this->innerController
      ->expects($this->once())
      ->method('editForm')
      ->with($entity)
      ->willReturn($expected_form);

    $result = $this->controller->editForm($entity);

    // Verify the form is returned with parallel marker.
    $this->assertIsArray($result);
    $this->assertTrue($result['#wse_parallel_relaxed'] ?? FALSE);
  }

  /**
   * Tests that relaxation is applied to revisionable entities.
   *
   * @covers ::relaxWorkspaceConstraints
   */
  public function testRelaxationForRevisionableEntity() {
    $entity = $this->createMockEntity('node', TRUE, FALSE);
    $form = ['#form_id' => 'node_edit_form', '#entity' => $entity];

    $this->innerController
      ->expects($this->once())
      ->method('editForm')
      ->with($entity)
      ->willReturn($form);

    $result = $this->controller->editForm($entity);

    // Verify parallel marker is added.
    $this->assertTrue($result['#wse_parallel_relaxed']);
  }

  /**
   * Tests that relaxation is NOT applied to non-revisionable entities.
   *
   * @covers ::relaxWorkspaceConstraints
   */
  public function testNoRelaxationForNonRevisionableEntity() {
    $entity = $this->createMockEntity('user', FALSE, FALSE);
    $form = ['#form_id' => 'user_edit_form', '#entity' => $entity];

    $this->innerController
      ->expects($this->once())
      ->method('editForm')
      ->with($entity)
      ->willReturn($form);

    $result = $this->controller->editForm($entity);

    // Verify parallel marker is NOT added for non-revisionable entities.
    $this->assertArrayNotHasKey('#wse_parallel_relaxed', $result);
  }

  /**
   * Tests that menu entities are skipped for wse_menu compatibility.
   *
   * @covers ::relaxWorkspaceConstraints
   */
  public function testMenuEntitySkipped() {
    $entity = $this->createMockEntity('menu_link_content', TRUE, TRUE);
    $form = ['#form_id' => 'menu_link_content_edit_form', '#entity' => $entity];

    $this->innerController
      ->expects($this->once())
      ->method('editForm')
      ->with($entity)
      ->willReturn($form);

    $result = $this->controller->editForm($entity);

    // Verify parallel marker is NOT added for menu entities.
    $this->assertArrayNotHasKey('#wse_parallel_relaxed', $result);
  }

  /**
   * Tests addForm delegation.
   *
   * @covers ::addForm
   */
  public function testAddFormDelegation() {
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $expected_form = ['#form' => 'add_test'];

    $this->innerController
      ->expects($this->once())
      ->method('addForm')
      ->with($entity_type, 'article')
      ->willReturn($expected_form);

    $result = $this->controller->addForm($entity_type, 'article');

    $this->assertEquals($expected_form, $result);
  }

  /**
   * Tests editTitle delegation.
   *
   * @covers ::editTitle
   */
  public function testEditTitleDelegation() {
    $entity = $this->createMockEntity('node', TRUE, FALSE);
    $expected_title = 'Edit Test Node';

    $this->innerController
      ->expects($this->once())
      ->method('editTitle')
      ->with($entity)
      ->willReturn($expected_title);

    $result = $this->controller->editTitle($entity);

    $this->assertEquals($expected_title, $result);
  }

  /**
   * Creates a mock entity for testing.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param bool $is_revisionable
   *   Whether the entity is revisionable.
   * @param bool $is_menu_entity
   *   Whether this is a menu-related entity.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked entity.
   */
  protected function createMockEntity($entity_type_id, $is_revisionable, $is_menu_entity) {
    $entity = $this->createMock(ContentEntityInterface::class);
    
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->method('isRevisionable')->willReturn($is_revisionable);
    
    $entity->method('getEntityType')->willReturn($entity_type);
    $entity->method('getEntityTypeId')->willReturn($entity_type_id);
    
    // Mock getTypedData for constraint testing.
    $typed_data = $this->createMock('Drupal\Core\TypedData\ComplexDataInterface');
    $typed_data->method('getConstraints')->willReturn([]);
    $entity->method('getTypedData')->willReturn($typed_data);

    return $entity;
  }

}
