<?php

namespace Drupal\Tests\wse_parallel\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\Insert;
use Drupal\Core\Database\Query\Select;
use Drupal\Core\Database\Query\Update;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\workspaces\WorkspaceInterface;
use Drupal\wse_parallel\Parallel\SessionTracker;
use Psr\Log\LoggerInterface;

/**
 * Tests the SessionTracker service.
 *
 * @coversDefaultClass \Drupal\wse_parallel\Parallel\SessionTracker
 * @group wse_parallel
 */
class SessionTrackerTest extends UnitTestCase {

  /**
   * The session tracker under test.
   *
   * @var \Drupal\wse_parallel\Parallel\SessionTracker
   */
  protected $sessionTracker;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The mocked logger.
   *
   * @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->sessionTracker = new SessionTracker($this->database, $this->logger);
  }

  /**
   * Tests that anonymous users are skipped.
   *
   * @covers ::startOrUpdateSession
   */
  public function testAnonymousUserSkipped() {
    $entity = $this->createMockEntity();
    $workspace = $this->createMock(WorkspaceInterface::class);
    $account = $this->createMock(AccountInterface::class);
    $account->method('isAnonymous')->willReturn(TRUE);

    $this->database->expects($this->never())->method('select');
    $this->database->expects($this->never())->method('insert');

    $result = $this->sessionTracker->startOrUpdateSession(
      $entity,
      1,
      1,
      $workspace,
      $account
    );

    $this->assertEquals(0, $result);
  }

  /**
   * Tests that non-revisionable entities are skipped.
   *
   * @covers ::startOrUpdateSession
   */
  public function testNonRevisionableEntitySkipped() {
    $entity = $this->createMockEntity(FALSE);
    $workspace = $this->createMock(WorkspaceInterface::class);
    $account = $this->createMock(AccountInterface::class);
    $account->method('isAnonymous')->willReturn(FALSE);
    $account->method('id')->willReturn(1);

    $this->database->expects($this->never())->method('select');
    $this->database->expects($this->never())->method('insert');

    $result = $this->sessionTracker->startOrUpdateSession(
      $entity,
      1,
      1,
      $workspace,
      $account
    );

    $this->assertEquals(0, $result);
  }

  /**
   * Tests creating a new session.
   *
   * @covers ::startOrUpdateSession
   */
  public function testCreateNewSession() {
    $entity = $this->createMockEntity();
    $workspace = $this->createMock(WorkspaceInterface::class);
    $workspace->method('id')->willReturn('stage');
    
    $account = $this->createMock(AccountInterface::class);
    $account->method('isAnonymous')->willReturn(FALSE);
    $account->method('id')->willReturn(1);

    // Mock the select query to return no existing session.
    $select = $this->createMock(Select::class);
    $select->method('fields')->willReturnSelf();
    $select->method('condition')->willReturnSelf();
    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchField')->willReturn(FALSE);
    $select->method('execute')->willReturn($statement);

    $this->database->method('select')->willReturn($select);

    // Mock the insert query.
    $insert = $this->createMock(Insert::class);
    $insert->method('fields')->willReturnSelf();
    $insert->method('execute')->willReturn(123);

    $this->database->method('insert')->willReturn($insert);

    $result = $this->sessionTracker->startOrUpdateSession(
      $entity,
      1,
      2,
      $workspace,
      $account
    );

    $this->assertEquals(123, $result);
  }

  /**
   * Tests closing sessions.
   *
   * @covers ::closeSessions
   */
  public function testCloseSessions() {
    $entity = $this->createMockEntity();
    $account = $this->createMock(AccountInterface::class);
    $account->method('isAnonymous')->willReturn(FALSE);
    $account->method('id')->willReturn(1);

    $update = $this->createMock(Update::class);
    $update->method('fields')->willReturnSelf();
    $update->method('condition')->willReturnSelf();
    $update->method('execute')->willReturn(2);

    $this->database->method('update')->willReturn($update);

    $this->sessionTracker->closeSessions($entity, $account);

    // If we get here without exceptions, the test passes.
    $this->assertTrue(TRUE);
  }

  /**
   * Creates a mock entity for testing.
   *
   * @param bool $is_revisionable
   *   Whether the entity is revisionable.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked entity.
   */
  protected function createMockEntity($is_revisionable = TRUE) {
    $entity = $this->createMock(ContentEntityInterface::class);
    
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->method('isRevisionable')->willReturn($is_revisionable);
    
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getId')->willReturn('en');
    
    $entity->method('getEntityType')->willReturn($entity_type);
    $entity->method('getEntityTypeId')->willReturn('node');
    $entity->method('id')->willReturn(1);
    $entity->method('getRevisionId')->willReturn(1);
    $entity->method('language')->willReturn($language);

    return $entity;
  }

}
