<?php

namespace Drupal\Tests\wse_parallel\Unit\ValueObject;

use Drupal\Tests\UnitTestCase;
use Drupal\wse_parallel\ValueObject\ParallelStateDto;

/**
 * Tests the ParallelStateDto value object.
 *
 * @coversDefaultClass \Drupal\wse_parallel\ValueObject\ParallelStateDto
 * @group wse_parallel
 */
class ParallelStateDtoTest extends UnitTestCase {

  /**
   * Tests basic getters.
   *
   * @covers ::__construct
   * @covers ::hasNewerPublish
   * @covers ::getActiveSessions
   * @covers ::getLastPublished
   * @covers ::hasDiverged
   * @covers ::getSuggestedAction
   */
  public function testBasicGetters() {
    $sessions = [
      ['sid' => 1, 'uid' => 2, 'workspace_id' => 'stage'],
    ];
    $lastPublished = [
      'plid' => 1,
      'to_revision_id' => 100,
      'published' => 1234567890,
    ];

    $dto = new ParallelStateDto(
      TRUE,
      $sessions,
      $lastPublished,
      TRUE,
      'conflict'
    );

    $this->assertTrue($dto->hasNewerPublish());
    $this->assertEquals($sessions, $dto->getActiveSessions());
    $this->assertEquals($lastPublished, $dto->getLastPublished());
    $this->assertTrue($dto->hasDiverged());
    $this->assertEquals('conflict', $dto->getSuggestedAction());
  }

  /**
   * Tests session count methods.
   *
   * @covers ::getActiveSessionCount
   * @covers ::hasActiveSessions
   */
  public function testSessionCounts() {
    $sessions = [
      ['sid' => 1],
      ['sid' => 2],
    ];

    $dto = new ParallelStateDto(FALSE, $sessions, NULL, FALSE, 'none');

    $this->assertEquals(2, $dto->getActiveSessionCount());
    $this->assertTrue($dto->hasActiveSessions());

    $emptyDto = new ParallelStateDto(FALSE, [], NULL, FALSE, 'none');
    $this->assertEquals(0, $emptyDto->getActiveSessionCount());
    $this->assertFalse($emptyDto->hasActiveSessions());
  }

  /**
   * Tests publish status methods.
   *
   * @covers ::hasBeenPublished
   */
  public function testPublishStatus() {
    $withPublish = new ParallelStateDto(
      FALSE,
      [],
      ['plid' => 1],
      FALSE,
      'none'
    );
    $this->assertTrue($withPublish->hasBeenPublished());

    $withoutPublish = new ParallelStateDto(FALSE, [], NULL, FALSE, 'none');
    $this->assertFalse($withoutPublish->hasBeenPublished());
  }

  /**
   * Tests parallel activity detection.
   *
   * @covers ::hasParallelActivity
   */
  public function testHasParallelActivity() {
    // Has newer publish.
    $dto1 = new ParallelStateDto(TRUE, [], NULL, FALSE, 'review');
    $this->assertTrue($dto1->hasParallelActivity());

    // Has active sessions.
    $dto2 = new ParallelStateDto(FALSE, [['sid' => 1]], NULL, FALSE, 'review');
    $this->assertTrue($dto2->hasParallelActivity());

    // Has diverged.
    $dto3 = new ParallelStateDto(FALSE, [], NULL, TRUE, 'merge');
    $this->assertTrue($dto3->hasParallelActivity());

    // No activity.
    $dto4 = new ParallelStateDto(FALSE, [], NULL, FALSE, 'none');
    $this->assertFalse($dto4->hasParallelActivity());
  }

  /**
   * Tests toArray method.
   *
   * @covers ::toArray
   */
  public function testToArray() {
    $sessions = [['sid' => 1]];
    $lastPublished = ['plid' => 1];

    $dto = new ParallelStateDto(
      TRUE,
      $sessions,
      $lastPublished,
      TRUE,
      'conflict'
    );

    $array = $dto->toArray();

    $this->assertIsArray($array);
    $this->assertTrue($array['has_newer_publish']);
    $this->assertEquals(1, $array['active_session_count']);
    $this->assertTrue($array['has_diverged']);
    $this->assertEquals('conflict', $array['suggested_action']);
    $this->assertTrue($array['has_parallel_activity']);
    $this->assertEquals($lastPublished, $array['last_published']);
  }

}
