<?php

/**
 * @file
 * Batch callbacks for WSE Parallel module.
 */

use Drupal\Core\Database\Database;

/**
 * DISABLED: Batch operation to create workspace_published_revisions for closed workspaces.
 *
 * This batch operation attempts to retroactively create workspace_published_revisions
 * records for closed workspaces that don't have them. It does this by:
 * 1. Finding all closed workspaces without workspace_published_revisions records
 * 2. For each workspace, querying revision tables to find the "newest" revision
 *    created at or before the workspace's 'changed' timestamp
 * 3. Creating workspace_published_revisions entries with those revision IDs
 *
 * REASON FOR DISABLING:
 * This approach has limitations and may not accurately reconstruct historical data:
 * - Relies on workspace.changed timestamp being accurate, but some workspaces have
 *   artificial timestamps (e.g., 1680195008 = March 30, 2023) that don't reflect
 *   when content was actually created
 * - For workspaces with incorrect timestamps, the query finds zero revisions and
 *   silently skips creating records (line 137: if (!empty($published_revision_ids)))
 * - Cannot reliably distinguish which revisions were actually in the workspace
 *   vs. which just happened to exist at that timestamp
 * - May produce incorrect historical records that could mislead users
 *
 * ALTERNATIVE APPROACH:
 * Instead of trying to reconstruct missing history, we've decided to:
 * - Accept that some closed workspaces will have incomplete historical data
 * - Focus on ensuring all FUTURE workspace publishes are properly logged
 * - Display closed workspaces without records gracefully in the UI
 * - Keep this code commented for reference in case requirements change
 *
 * @param array $context
 *   The batch context.
 */
/*
function wse_parallel_create_closed_workspace_records_batch_operation(&$context) {
  $connection = Database::getConnection();
  $entity_type_manager = \Drupal::entityTypeManager();
  $workspace_manager = \Drupal::service('workspaces.manager');

  // Initialize batch sandbox on first iteration.
  if (!isset($context['sandbox']['progress'])) {
    $context['sandbox']['progress'] = 0;
    $context['sandbox']['current_workspace'] = 0;
    $context['sandbox']['processed_revisions'] = [];

    // Get all closed workspaces that don't already have workspace_published_revisions records.
    $query = $connection->select('workspace', 'w');
    $query->fields('w', ['id', 'changed', 'label']);
    $query->leftJoin('workspace_published_revisions', 'wpr', 'w.id = wpr.workspace_id');
    $query->condition('w.status', 'closed');
    $query->isNull('wpr.workspace_id');
    $query->orderBy('w.changed', 'ASC');
    $query->orderBy('w.created', 'ASC');

    $context['sandbox']['workspaces'] = $query->execute()->fetchAllAssoc('id');

    $context['sandbox']['max'] = count($context['sandbox']['workspaces']);
    $context['results']['created'] = 0;
  }

  if (empty($context['sandbox']['workspaces'])) {
    $context['finished'] = 1;
    return;
  }

  // Process one workspace per batch iteration.
  $workspaces_array = array_values($context['sandbox']['workspaces']);
  $workspace = $workspaces_array[$context['sandbox']['current_workspace']];

  $published_revision_ids = [];
  $revert_revision_ids = [];
  $is_first_workspace = ($context['sandbox']['current_workspace'] === 0);

  // Get all workspace-supported entity types.
  foreach ($entity_type_manager->getDefinitions() as $entity_type_id => $entity_type) {
    if (!$workspace_manager->isEntityTypeSupported($entity_type)) {
      continue;
    }

    $storage = $entity_type_manager->getStorage($entity_type_id);
    $revision_table = $entity_type->getRevisionTable();

    if (empty($revision_table)) {
      continue;
    }

    $table_mapping = $storage->getTableMapping();
    $id_field = $table_mapping->getColumnNames($entity_type->getKey('id'))['value'];
    $revision_id_field = $table_mapping->getColumnNames($entity_type->getKey('revision'))['value'];

    // Get the revision_created field from revision metadata keys.
    $revision_metadata_keys = $entity_type->get('revision_metadata_keys');
    if (empty($revision_metadata_keys['revision_created'])) {
      // Skip entity types without revision_created metadata.
      continue;
    }
    $revision_created_field = $table_mapping->getColumnNames($revision_metadata_keys['revision_created'])['value'];

    // Subquery to find max created time AND max revision_id per entity.
    $subquery = $connection->select($revision_table, 'r2');
    $subquery->addField('r2', $id_field);
    $subquery->addExpression("MAX($revision_created_field)", 'max_created');
    $subquery->addExpression("MAX($revision_id_field)", 'max_revision_id');
    $subquery->condition($revision_created_field, $workspace->changed, '<=');
    $subquery->groupBy($id_field);

    // Main query joins on BOTH conditions.
    $query = $connection->select($revision_table, 'r');
    $query->addField('r', $id_field, 'entity_id');
    $query->addField('r', $revision_id_field, 'revision_id');
    $query->join($subquery, 'max_r',
      "r.$id_field = max_r.$id_field AND r.$revision_id_field = max_r.max_revision_id"
    );

    $results = $query->execute()->fetchAll();

    foreach ($results as $row) {
      $entity_id = $row->entity_id;
      $revision_id = $row->revision_id;

      // Check if we've already processed this entity with this revision.
      $previous_revision = $context['sandbox']['processed_revisions'][$entity_type_id][$entity_id] ?? NULL;

      if ($previous_revision === $revision_id) {
        // Same revision as before, skip it.
        continue;
      }

      // Add to published_revision_ids.
      $published_revision_ids[$entity_type_id][$revision_id] = $entity_id;

      // For revert_revision_ids, use the previous revision or find the oldest.
      if ($is_first_workspace || $previous_revision === NULL) {
        // Find the oldest revision for this entity.
        $oldest_revision = $connection->select($revision_table, 'r')
          ->fields('r', [$revision_id_field])
          ->condition($id_field, $entity_id)
          ->orderBy($revision_created_field, 'ASC')
          ->range(0, 1)
          ->execute()
          ->fetchField();

        $revert_revision_ids[$entity_type_id][$oldest_revision] = $entity_id;
      }
      else {
        // Use the previous revision from our tracking.
        $revert_revision_ids[$entity_type_id][$previous_revision] = $entity_id;
      }

      // Update our tracking.
      $context['sandbox']['processed_revisions'][$entity_type_id][$entity_id] = $revision_id;
    }
  }

  // Create the workspace_published_revisions record.
  if (!empty($published_revision_ids)) {
    $connection->insert('workspace_published_revisions')
      ->fields([
        'workspace_id' => $workspace->id,
        'workspace_label' => $workspace->label,
        'published_revision_ids' => json_encode($published_revision_ids),
        'revert_revision_ids' => json_encode($revert_revision_ids),
        'published_on' => $workspace->changed,
      ])
      ->execute();

    $context['results']['created']++;
  }

  $context['sandbox']['current_workspace']++;
  $context['sandbox']['progress']++;

  // Update progress message.
  $context['message'] = t('Processing closed workspace @current of @total', [
    '@current' => $context['sandbox']['progress'],
    '@total' => $context['sandbox']['max'],
  ]);

  // Inform the batch engine of our progress.
  if ($context['sandbox']['progress'] < $context['sandbox']['max']) {
    $context['finished'] = $context['sandbox']['progress'] / $context['sandbox']['max'];
  }
  else {
    $context['finished'] = 1;
  }
}
*/

/**
 * Batch operation callback for backfilling publish log.
 *
 * @param array $context
 *   The batch context.
 */
function wse_parallel_backfill_batch_operation(&$context) {
  $connection = Database::getConnection();
  $entity_type_manager = \Drupal::entityTypeManager();

  // Initialize batch sandbox on first iteration.
  if (!isset($context['sandbox']['progress'])) {
    $context['sandbox']['progress'] = 0;
    $context['sandbox']['offset'] = 0;
    $context['sandbox']['max'] = $connection->select('workspace_published_revisions', 'wpr')
      ->countQuery()
      ->execute()
      ->fetchField();
    $context['results']['inserted'] = 0;

    // Initialize snapshot_data with all default revisions before the first workspace publish.
    $context['sandbox']['snapshot_data'] = [];

    // Get the timestamp of the first workspace publish.
    $first_publish_timestamp = $connection->select('workspace_published_revisions', 'wpr')
      ->fields('wpr', ['published_on'])
      ->orderBy('published_on', 'ASC')
      ->range(0, 1)
      ->execute()
      ->fetchField();

    if ($first_publish_timestamp) {
      $workspace_manager = \Drupal::service('workspaces.manager');

      // Get all workspace-supported entity types.
      foreach ($entity_type_manager->getDefinitions() as $entity_type_id => $entity_type) {
        if (!$workspace_manager->isEntityTypeSupported($entity_type)) {
          continue;
        }

        $storage = $entity_type_manager->getStorage($entity_type_id);
        $revision_table = $entity_type->getRevisionTable();

        if (empty($revision_table)) {
          continue;
        }

        $table_mapping = $storage->getTableMapping();
        $id_field = $table_mapping->getColumnNames($entity_type->getKey('id'))['value'];
        $revision_id_field = $table_mapping->getColumnNames($entity_type->getKey('revision'))['value'];

        // Get the revision_created field from revision metadata keys.
        $revision_metadata_keys = $entity_type->get('revision_metadata_keys');
        if (empty($revision_metadata_keys['revision_created'])) {
          // Skip entity types without revision_created metadata.
          continue;
        }
        $revision_created_field = $table_mapping->getColumnNames($revision_metadata_keys['revision_created'])['value'];

        // Subquery to find max created time AND max revision_id per entity
        // for revisions created BEFORE the first workspace publish.
        $subquery = $connection->select($revision_table, 'r2');
        $subquery->addField('r2', $id_field);
        $subquery->addExpression("MAX($revision_created_field)", 'max_created');
        $subquery->addExpression("MAX($revision_id_field)", 'max_revision_id');
        $subquery->condition($revision_created_field, $first_publish_timestamp, '<');
        $subquery->groupBy($id_field);

        // Main query joins on BOTH conditions.
        $query = $connection->select($revision_table, 'r');
        $query->addField('r', $id_field, 'entity_id');
        $query->addField('r', $revision_id_field, 'revision_id');
        $query->join($subquery, 'max_r',
          "r.$id_field = max_r.$id_field AND r.$revision_id_field = max_r.max_revision_id"
        );

        $results = $query->execute()->fetchAll();

        foreach ($results as $row) {
          if (!isset($context['sandbox']['snapshot_data'][$entity_type_id])) {
            $context['sandbox']['snapshot_data'][$entity_type_id] = [];
          }
          $context['sandbox']['snapshot_data'][$entity_type_id][$row->entity_id] = $row->revision_id;
        }
      }
    }
  }

  // Process workspaces in batches of 10.
  $limit = 10;
  $published_workspaces = $connection->select('workspace_published_revisions', 'wpr')
    ->fields('wpr')
    ->orderBy('published_on', 'ASC')
    ->range($context['sandbox']['offset'], $limit)
    ->execute()
    ->fetchAll();

  foreach ($published_workspaces as $workspace_publish) {
    $published_revisions = json_decode($workspace_publish->published_revision_ids, TRUE);
    $revert_revisions = json_decode($workspace_publish->revert_revision_ids, TRUE);

    if (empty($published_revisions) || !is_array($published_revisions)) {
      $context['sandbox']['progress']++;
      continue;
    }

    foreach ($published_revisions as $entity_type => $revisions) {
      // Skip entity types that don't exist (e.g., wse_menu_tree).
      if (!$entity_type_manager->hasDefinition($entity_type)) {
        continue;
      }

      $storage = $entity_type_manager->getStorage($entity_type);
      $entity_type_def = $storage->getEntityType();
      $revision_table = $entity_type_def->getRevisionTable();

      if (empty($revision_table)) {
        continue;
      }

      $table_mapping = $storage->getTableMapping();
      $revision_id_field = $table_mapping->getColumnNames($entity_type_def->getKey('revision'))['value'];
      $langcode_field = $table_mapping->getColumnNames($entity_type_def->getKey('langcode'))['value'];

      // Load langcodes for all revisions in bulk.
      $revision_ids = array_keys($revisions);
      $langcode_map = [];
      if (!empty($revision_ids)) {
        $result = $connection->select($revision_table, 'r')
          ->fields('r', [$revision_id_field, $langcode_field])
          ->condition($revision_id_field, $revision_ids, 'IN')
          ->execute();

        foreach ($result as $row) {
          $langcode_map[$row->{$revision_id_field}] = $row->{$langcode_field};
        }
      }

      foreach ($revisions as $to_revision_id => $entity_id) {
        $from_revision_id = NULL;
        if (isset($revert_revisions[$entity_type]) && is_array($revert_revisions[$entity_type])) {
          $from_revision_id = array_search($entity_id, $revert_revisions[$entity_type]);
        }

        if ($from_revision_id === FALSE || $from_revision_id === NULL) {
          throw new \Exception(sprintf(
            'Failed to find matching from_revision_id for entity %s:%s in workspace %s during publish log backfill.',
            $entity_type,
            $entity_id,
            $workspace_publish->workspace_id
          ));
        }

        $langcode = $langcode_map[$to_revision_id] ?? NULL;

        $connection->insert('wse_parallel_publish_log')
          ->fields([
            'entity_type' => $entity_type,
            'entity_id' => $entity_id,
            'langcode' => $langcode,
            'from_revision_id' => $from_revision_id,
            'to_revision_id' => $to_revision_id,
            'workspace_id' => $workspace_publish->workspace_id,
            'published' => $workspace_publish->published_on,
          ])
          ->execute();

        $context['results']['inserted']++;
      }
    }

    // Create the snapshot for this workspace.
    // Start with the current snapshot state (from previous workspaces).
    $snapshot_data = $context['sandbox']['snapshot_data'];

    // Merge in this workspace's published revisions.
    foreach ($published_revisions as $entity_type => $revisions) {
      if (!isset($snapshot_data[$entity_type])) {
        $snapshot_data[$entity_type] = [];
      }
      foreach ($revisions as $revision_id => $entity_id) {
        $snapshot_data[$entity_type][$entity_id] = $revision_id;
      }
    }

    // Save the snapshot.
    if (!empty($snapshot_data)) {
      $connection->insert('wse_parallel_workspace_snapshot')
        ->fields([
          'workspace_id' => $workspace_publish->workspace_id,
          'snapshot_data' => json_encode($snapshot_data),
          'published_timestamp' => $workspace_publish->published_on,
        ])
        ->execute();

      // Update the sandbox for the next workspace.
      $context['sandbox']['snapshot_data'] = $snapshot_data;
    }

    $context['sandbox']['progress']++;
  }

  // Update offset for next iteration.
  $context['sandbox']['offset'] += $limit;

  // Update progress message.
  $context['message'] = t('Processing workspace @current of @total', [
    '@current' => $context['sandbox']['progress'],
    '@total' => $context['sandbox']['max'],
  ]);

  // Inform the batch engine that we are not finished yet.
  if ($context['sandbox']['progress'] < $context['sandbox']['max']) {
    $context['finished'] = $context['sandbox']['progress'] / $context['sandbox']['max'];
  }
  else {
    $context['finished'] = 1;
  }
}

/**
 * Batch finished callback for backfilling publish log.
 *
 * @param bool $success
 *   TRUE if the batch completed successfully.
 * @param array $results
 *   The results array.
 * @param array $operations
 *   The operations array.
 */
function wse_parallel_backfill_batch_finished($success, $results, $operations) {
  if ($success) {
    $inserted = $results['inserted'] ?? 0;
    \Drupal::messenger()->addStatus(t('Successfully backfilled @count publish log entries from historic workspace data.', [
      '@count' => $inserted,
    ]));
  }
  else {
    \Drupal::messenger()->addError(t('An error occurred while backfilling the publish log.'));
  }
}
