<?php

declare(strict_types=1);

namespace Drupal\wse_task_monitor;

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Enumeration of task statuses.
 */
enum TaskStatus: string {

  case Pending = 'pending';
  case Running = 'running';
  case Completed = 'completed';
  case Failed = 'failed';
  case Cancelled = 'cancelled';

  /**
   * Checks if the task status represents a finished state.
   *
   * @return bool
   *   TRUE if the task is finished (completed, failed, or cancelled).
   */
  public function isFinished(): bool {
    return in_array($this, [
      self::Completed,
      self::Failed,
      self::Cancelled,
    ], TRUE);
  }

  /**
   * Gets the human-readable label for this status.
   *
   * @return string
   *   The translated status label.
   */
  public function getLabel(): string {
    return match($this) {
      self::Pending => (string) new TranslatableMarkup('Pending'),
      self::Running => (string) new TranslatableMarkup('Running'),
      self::Completed => (string) new TranslatableMarkup('Completed'),
      self::Failed => (string) new TranslatableMarkup('Failed'),
      self::Cancelled => (string) new TranslatableMarkup('Cancelled'),
    };
  }

  /**
   * Check if a task data array has this status.
   *
   * @param array $task_data
   *   The task data array containing a 'status' key.
   *
   * @return bool
   *   TRUE if the task has this status.
   */
  public function matches(array $task_data): bool {
    return ($task_data['status'] ?? '') === $this->value;
  }

}
