<?php

namespace Drupal\wse_scheduler;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Utility\Error;
use Drupal\workspaces\WorkspaceTrackerInterface;
use Drupal\wse_scheduler\Event\WorkspaceScheduledPublishEvent;
use Drupal\wse_scheduler\Event\WorkspaceSchedulerEvents;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * The scheduled workspace publisher.
 */
class ScheduledWorkspacePublisher {

  use LoggerChannelTrait;

  public function __construct(
    protected EventDispatcherInterface $eventDispatcher,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected TimeInterface $time,
    protected WorkspaceTrackerInterface $workspaceTracker,
  ) {}

  /**
   * Publishes due workspaces scheduled for publishing.
   */
  public function publishScheduledWorkspaces() {
    $logger = $this->getLogger('wse_scheduler');
    try {
      /** @var \Drupal\workspaces\WorkspaceInterface $workspace */
      foreach ($this->loadScheduledWorkspaces() as $workspace) {
        $tracked_entities = $this->workspaceTracker->getTrackedEntities($workspace->id());

        /** @var \Drupal\wse_scheduler\Event\WorkspaceScheduledPublishEvent $event */
        $event = $this->eventDispatcher->dispatch(
          new WorkspaceScheduledPublishEvent($workspace, $tracked_entities),
          WorkspaceSchedulerEvents::SCHEDULED_PUBLISHING_ELIGIBLE
        );

        if ($event->isPublishingEligible()) {
          $workspace->publish();
          $logger->notice(
            'Workspace @workspace scheduled for publishing was successfully published.',
            ['@workspace' => $workspace->label()]
          );
        }
        else {
          $logger->error(
            'Workspace @workspace scheduled for publishing was not eligible for publishing.',
            ['@workspace' => $workspace->label()]
          );
        }
      }
    }
    catch (\Exception $exception) {
      Error::logException($logger, $exception);
    }
  }

  /**
   * Loads workspaces which were scheduled and are due for publishing.
   *
   * @return \Drupal\workspaces\WorkspaceInterface[]
   *   An array of scheduled workspaces.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function loadScheduledWorkspaces() {
    $storage = $this->entityTypeManager->getStorage('workspace');
    $result = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('status', WSE_STATUS_OPEN)
      ->condition('publish_on', $this->time->getCurrentTime(), '<')
      ->execute();

    return $result ? $storage->loadMultiple($result) : [];
  }

}
