<?php

declare(strict_types=1);

namespace Drupal\wse\EventSubscriber;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\workspaces\Event\WorkspaceSwitchEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Handles workspace switch events.
 */
class WorkspaceSwitchSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected RequestStack $requestStack,
    protected PrivateTempStoreFactory $tempStoreFactory,
    protected TimeInterface $time,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      WorkspaceSwitchEvent::class => 'onWorkspaceSwitch',
    ];
  }

  /**
   * Handles workspace switch events.
   */
  public function onWorkspaceSwitch(WorkspaceSwitchEvent $event): void {
    $this->removeWorkspaceQueryParam();

    // Record recently activated workspaces.
    $workspace = $event->getWorkspace();
    if ($workspace) {
      $temp_store = $this->tempStoreFactory->get('wse');
      $recent_workspaces = $temp_store->get('recent_workspaces') ?: [];
      $recent_workspaces[$workspace->id()] = $this->time->getRequestTime();
      $temp_store->set('recent_workspaces', $recent_workspaces);
    }
  }

  /**
   * Removes any trace of the current workspace from the query parameters.
   */
  private function removeWorkspaceQueryParam(): void {
    // When switching workspaces, ensure that there's no workspace query
    // parameter, either standalone or in the destination.
    $request = $this->requestStack->getCurrentRequest();
    if (!$request) {
      return;
    }

    $request->query->remove('workspace');

    $query_params = $request->query->all();
    if (isset($query_params['destination'])) {
      $destination = UrlHelper::parse($query_params['destination']);
      unset($destination['query']['workspace']);
      $new_destination = $destination['path'] . ($destination['query'] ? ('?' . UrlHelper::buildQuery($destination['query'])) : '');
      $request->query->set('destination', $new_destination);
    }
  }

}
