<?php

namespace Drupal\wse\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\WorkspaceSafeFormInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\workspaces\WorkspaceInterface;
use Drupal\workspaces\WorkspaceTrackerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the form for discarding the changes to an entity in a workspace.
 */
class DiscardEntityForm extends ConfirmFormBase implements WorkspaceSafeFormInterface, ContainerInjectionInterface {

  use EntityDependencyTrait;

  /**
   * The entity that will be discarded.
   *
   * @var \Drupal\Core\Entity\RevisionableInterface
   */
  protected RevisionableInterface $entity;

  /**
   * The workspace from which the entity will be discarded.
   *
   * @var \Drupal\workspaces\WorkspaceInterface
   */
  protected WorkspaceInterface $sourceWorkspace;

  public function __construct(
    protected WorkspaceTrackerInterface $workspaceTracker,
    protected QueueFactory $queueFactory,
    protected TimeInterface $time,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('workspaces.tracker'),
      $container->get('queue'),
      $container->get('datetime.time'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'workspace_discard_entity_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?RouteMatchInterface $route_match = NULL, ?WorkspaceInterface $source_workspace = NULL) {
    $this->entity = $route_match->getParameter($route_match->getParameter('entity_type_id'));
    $this->sourceWorkspace = $source_workspace;

    $form = parent::buildForm($form, $form_state);

    $this->addDependenciesFormElements($form, $form_state, $this->entity, $this->sourceWorkspace);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Build list of entities to discard.
    $affected_entity_ids[$this->entity->getEntityTypeId()][$this->entity->id()] = TRUE;
    $affected_entity_ids = array_replace_recursive($affected_entity_ids, $this->getSelectedDependencies($form_state));

    try {
      $workspace_id = $this->sourceWorkspace->id();
      $queue = $this->queueFactory->get('wse_revision_cleaner');

      foreach ($affected_entity_ids as $entity_type_id => $entity_ids) {
        $entity_id_list = array_keys($entity_ids);

        // Get all tracked revisions for the entities.
        $tracked_revisions = $this->workspaceTracker->getAllTrackedRevisions($workspace_id, $entity_type_id, $entity_id_list);

        // Get initial revisions (entities created in this workspace).
        $initial_revisions = $this->workspaceTracker->getTrackedInitialRevisions($workspace_id, $entity_type_id, $entity_id_list);

        // Remove association entries.
        $this->workspaceTracker->deleteTrackedEntities($workspace_id, $entity_type_id, $entity_id_list);

        // Create queue item for revision cleanup.
        if ($tracked_revisions) {
          $queue->createItem([
            'process_time' => $this->time->getCurrentTime(),
            'entity_type_id' => $entity_type_id,
            'items' => array_keys($tracked_revisions),
            'initial_revision_ids' => array_keys($initial_revisions),
          ]);
        }
      }

      $this->messenger()->addStatus($this->t('Successful operation.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Operation failed. All errors have been logged.'));
      $this->getLogger('workspaces')->error($e->getMessage());
    }

    $form_state->setRedirectUrl($this->sourceWorkspace->toUrl());
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Discard changes for @entity?', [
      '@entity' => $this->entity->label(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('All the revisions that were created in this workspace will be deleted. <strong>This action can not be undone!</strong>');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return $this->sourceWorkspace->toUrl();
  }

}
