<?php

namespace Drupal\wse\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\depcalc\DependencyStack;
use Drupal\depcalc\DependentEntityWrapper;
use Drupal\depcalc\DependentEntityWrapperInterface;
use Drupal\workspaces\WorkspaceInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Drupal\workspaces\WorkspaceTrackerInterface;

/**
 * Provides methods for handling entity dependencies in forms.
 */
trait EntityDependencyTrait {

  /**
   * Gathers all affected dependencies of an entity within a workspace.
   *
   * @param array $affected_entity_ids
   *   An array to collect affected entity IDs.
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity to gather dependencies for.
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The workspace context.
   */
  protected function gatherAffectedDependencies(array &$affected_entity_ids, RevisionableInterface $entity, WorkspaceInterface $workspace): void {
    $dependencies = $this->getWorkspaceManager()->executeInWorkspace($workspace->id(), function () use ($entity) {
      $wrapper = new DependentEntityWrapper($entity);
      $dependency_stack = new DependencyStack();
      $dependency_stack->ignoreCache(TRUE);
      $dependency_stack->ignoreConfig(TRUE);

      return \Drupal::service('entity.dependency.calculator')->calculateDependencies($wrapper, $dependency_stack);
    });

    $tracked_entities = $this->getWorkspaceTracker()->getTrackedEntities($workspace->id());
    foreach ($dependencies as $dependency) {
      if ($dependency instanceof DependentEntityWrapperInterface
        && isset($tracked_entities[$dependency->getEntityTypeId()])
        && in_array($dependency->getId(), $tracked_entities[$dependency->getEntityTypeId()])
      ) {
        $affected_entity_ids[$dependency->getEntityTypeId()][$dependency->getId()] = $dependency->getEntity()->label();
      }
    }
  }

  /**
   * Adds the dependencies form elements.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity to gather dependencies for.
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The workspace context.
   */
  protected function addDependenciesFormElements(array &$form, FormStateInterface $form_state, RevisionableInterface $entity, WorkspaceInterface $workspace): void {
    if (!\Drupal::moduleHandler()->moduleExists('depcalc')) {
      return;
    }

    $form['include_dependencies'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include dependencies'),
      '#default_value' => FALSE,
      '#ajax' => [
        'callback' => '::updateDependenciesFieldset',
        'wrapper' => 'dependencies-wrapper',
        'event' => 'change',
      ],
    ];

    $form['dependencies_container'] = $this->buildDependenciesFieldset($form_state, $entity, $workspace);

    // Attach JavaScript library for modal repositioning.
    $form['#attached']['library'][] = 'wse/discard-entity-modal';
  }

  /**
   * AJAX callback for updating dependencies fieldset.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The updated form element.
   */
  public function updateDependenciesFieldset(array $form, FormStateInterface $form_state): array {
    return $form['dependencies_container'];
  }

  /**
   * Builds the dependencies fieldset.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\Core\Entity\RevisionableInterface $entity
   *   The entity to gather dependencies for.
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The workspace context.
   *
   * @return array
   *   The fieldset render array.
   */
  protected function buildDependenciesFieldset(FormStateInterface $form_state, RevisionableInterface $entity, WorkspaceInterface $workspace): array {
    // Create container wrapper for AJAX replacement.
    $container = [
      '#type' => 'container',
      '#attributes' => ['id' => 'dependencies-wrapper'],
      '#states' => [
        'visible' => [
          ':input[name="include_dependencies"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Only show dependencies if checkbox is checked.
    if (!$form_state->getValue('include_dependencies', FALSE)) {
      return $container;
    }

    $dependencies_data = [];
    $this->gatherAffectedDependencies($dependencies_data, $entity, $workspace);

    // Ensure that the entity itself is not part of the array.
    unset($dependencies_data[$entity->getEntityTypeId()][$entity->id()]);
    $dependencies_data = array_filter($dependencies_data);

    if (!$dependencies_data) {
      $container['no_dependencies'] = [
        '#markup' => '<p>' . $this->t('No dependencies found in this workspace.') . '</p>',
      ];
      return $container;
    }

    // Create separate details element for each entity type.
    foreach ($dependencies_data as $entity_type_id => $entities) {
      $entity_type = $this->getEntityTypeManager()->getDefinition($entity_type_id);
      $count = count($entities);

      $container['details_' . $entity_type_id] = [
        '#type' => 'details',
        '#title' => $entity_type->getCountLabel($count),
        '#open' => FALSE,
      ];

      $options = array_combine(
        array_map(fn($id) => "$entity_type_id:$id", array_keys($entities)),
        $entities
      );
      $container['details_' . $entity_type_id]['dependencies'] = [
        '#type' => 'checkboxes',
        '#options' => $options,
        '#default_value' => array_keys($options),
        '#validated' => TRUE,
        '#description' => $this->formatPlural($count, 'Select to discard this @type.', 'Select which @types to discard.', [
          '@type' => $entity_type->getSingularLabel(),
          '@types' => $entity_type->getPluralLabel(),
        ]),
      ];
    }

    return $container;
  }

  /**
   * Extracts selected dependencies from the form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   An array of entity IDs keyed by entity type ID.
   */
  protected function getSelectedDependencies(FormStateInterface $form_state): array {
    $dependencies = [];

    if ($form_state->getValue('include_dependencies')) {
      foreach ($form_state->getValue('dependencies', []) as $key) {
        if (str_contains($key, ':')) {
          [$entity_type_id, $entity_id] = explode(':', $key, 2);
          $dependencies[$entity_type_id][$entity_id] = TRUE;
        }
      }
    }

    return $dependencies;
  }

  /**
   * Gets the workspace tracker service.
   *
   * @return \Drupal\workspaces\WorkspaceTrackerInterface
   *   The workspace tracker service.
   */
  protected function getWorkspaceTracker(): WorkspaceTrackerInterface {
    return \Drupal::service('workspaces.tracker');
  }

  /**
   * Gets the workspace manager service.
   *
   * @return \Drupal\workspaces\WorkspaceManagerInterface
   *   The workspace manager service.
   */
  protected function getWorkspaceManager(): WorkspaceManagerInterface {
    return \Drupal::service('workspaces.manager');
  }

  /**
   * Gets the entity type manager service.
   *
   * @return \Drupal\Core\Entity\EntityTypeManagerInterface
   *   The entity type manager service.
   */
  protected function getEntityTypeManager(): EntityTypeManagerInterface {
    return \Drupal::service('entity_type.manager');
  }

}
