<?php

declare(strict_types=1);

namespace Drupal\Tests\wse\FunctionalJavascript;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Url;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\field\Traits\EntityReferenceFieldCreationTrait;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\Tests\wse\Functional\WseTestUtilities;
use Drupal\workspaces\Entity\Workspace;
use Drupal\workspaces\WorkspaceInterface;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the discard entity functionality.
 */
#[Group('wse')]
#[RunTestsInSeparateProcesses]
class DiscardEntityTest extends WebDriverTestBase {

  use ContentTypeCreationTrait;
  use EntityReferenceFieldCreationTrait;
  use NodeCreationTrait;
  use WseTestUtilities;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'depcalc',
    'field',
    'field_ui',
    'node',
    'workspaces',
    'workspaces_ui',
    'wse',
  ];

  /**
   * A test workspace.
   */
  protected WorkspaceInterface $workspace;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->drupalCreateContentType(['type' => 'article', 'label' => 'Article']);

    // Create an entity reference field.
    $this->createEntityReferenceField(
      'node',
      'article',
      'field_reference',
      'Reference',
      'node',
      'default',
      ['target_bundles' => ['article' => 'article']],
      FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED
    );

    $this->setupWorkspaceSwitcherBlock();

    $permissions = [
      'create workspace',
      'edit own workspace',
      'view own workspace',
      'bypass entity access own workspace',
      'create article content',
      'edit any article content',
      'delete any article content',
    ];
    $admin = $this->drupalCreateUser($permissions);
    $this->drupalLogin($admin);

    // Create a test workspace programmatically.
    $this->workspace = Workspace::create([
      'id' => 'test',
      'label' => 'Test',
    ]);
    $this->workspace->save();
    \Drupal::service('workspaces.manager')->setActiveWorkspace($this->workspace);
  }

  /**
   * Tests discarding entity without dependencies.
   */
  public function testDiscardEntityWithoutDependencies(): void {
    /** @var \Drupal\workspaces\WorkspaceTrackerInterface $workspace_tracker */
    $workspace_tracker = \Drupal::service('workspaces.tracker');

    // Create a node in the workspace.
    $node = $this->createNode(['title' => 'Test Node', 'type' => 'article']);

    // Verify the node is tracked in the workspace.
    $tracked = $workspace_tracker->getTrackedEntities($this->workspace->id());
    $this->assertContains($node->id(), $tracked['node']);

    // Navigate to discard form.
    $this->drupalGet(Url::fromRoute('entity.node.discard_changes', [
      'node' => $node->id(),
      'source_workspace' => $this->workspace->id(),
    ]));

    // Verify form elements.
    $this->assertSession()->checkboxNotChecked('include_dependencies');

    // Submit form.
    $this->getSession()->getPage()->pressButton('Confirm');

    // Reset the static cache of the workspace tracker.
    \Drupal::getContainer()->set('workspaces.tracker', NULL);
    /** @var \Drupal\workspaces\WorkspaceTrackerInterface $workspace_tracker */
    $workspace_tracker = \Drupal::service('workspaces.tracker');

    // Verify entity was discarded.
    $tracked_after = $workspace_tracker->getTrackedEntities($this->workspace->id());
    $this->assertEmpty($tracked_after);
  }

  /**
   * Tests discarding entity with dependencies and selective discarding.
   */
  public function testDiscardEntityWithDependencies(): void {
    /** @var \Drupal\workspaces\WorkspaceTrackerInterface $workspace_tracker */
    $workspace_tracker = \Drupal::service('workspaces.tracker');

    // Create node1 and node2 first (will be referenced by node3).
    $node1 = $this->createNode(['title' => 'Node 2', 'type' => 'article']);
    $node2 = $this->createNode(['title' => 'Node 3', 'type' => 'article']);

    // Create node3 with references to both node1 and node2.
    $node3 = $this->createNode([
      'title' => 'Node 1',
      'type' => 'article',
      'field_reference' => [
        ['target_id' => $node1->id()],
        ['target_id' => $node2->id()],
      ],
    ]);

    $tracked_revisions = [
      1 => 1,
      2 => 1,
      3 => 2,
      4 => 2,
      5 => 3,
      6 => 3,
    ];
    $this->assertEquals($tracked_revisions, $workspace_tracker->getAllTrackedRevisions($this->workspace->id(), 'node'));

    // Navigate to discard form for node3.
    $this->drupalGet(Url::fromRoute('entity.node.discard_changes', [
      'node' => $node3->id(),
      'source_workspace' => $this->workspace->id(),
    ]));

    $page = $this->getSession()->getPage();
    $assert_session = $this->assertSession();

    // Check the include_dependencies checkbox to show dependencies.
    $page->checkField('include_dependencies');
    $assert_session->assertExpectedAjaxRequest(1);

    // Find and open the details element containing the dependencies.
    $details = $page->find('css', '#dependencies-wrapper details');
    $this->assertNotNull($details, 'Details element should exist');
    $details->find('css', 'summary')->click();

    $checkbox = $page->findField('dependencies[node:' . $node1->id() . ']');
    $this->assertTrue($checkbox->isChecked(), 'Node 1 checkbox should exist');

    // Uncheck node2 dependency - we only want to discard node1 and node3.
    $checkbox = $page->findField('dependencies[node:' . $node2->id() . ']');
    $this->assertTrue($checkbox->isChecked(), 'Node 2 checkbox should exist');
    $checkbox->uncheck();

    // Submit the form.
    $page->pressButton('Confirm');

    // Reset the static cache of the workspace tracker.
    \Drupal::getContainer()->set('workspaces.tracker', NULL);
    /** @var \Drupal\workspaces\WorkspaceTrackerInterface $workspace_tracker */
    $workspace_tracker = \Drupal::service('workspaces.tracker');

    // Verify node1 and node3 were discarded, node2 should still be tracked.
    $tracked_entities = ['node' => [4 => 2]];
    $this->assertEquals($tracked_entities, $workspace_tracker->getTrackedEntities($this->workspace->id()));

    $tracked_revisions = [
      3 => 2,
      4 => 2,
    ];
    $this->assertEquals($tracked_revisions, $workspace_tracker->getAllTrackedRevisions($this->workspace->id(), 'node'));
  }

}
