<?php

declare(strict_types=1);

namespace Drupal\Tests\wse\Kernel;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\Tests\workspaces\Kernel\WorkspaceTestTrait;
use Drupal\workspaces\Entity\Workspace;
use Drupal\wse\WseWorkspaceManager;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests workspace purge logic.
 */
#[CoversClass(WseWorkspaceManager::class)]
#[Group('wse')]
#[RunTestsInSeparateProcesses]
class WseWorkspacePurgeTest extends KernelTestBase {

  use NodeCreationTrait;
  use WorkspaceTestTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'options',
    'node',
    'user',
    'workspaces',
    'wse',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installEntitySchema('user');
    $this->installEntitySchema('workspace');
    $this->installSchema('workspaces', ['workspace_association', 'workspace_association_revision']);
    $this->installSchema('wse', ['workspace_published_revisions']);
  }

  /**
   * Tests workspace purging.
   */
  public function testWorkspacePurging(): void {
    // Create an open workspace with a node inside it.
    $this->workspaces['open'] = Workspace::create(['id' => 'open', 'label' => 'Open']);
    $this->workspaces['open']->save();
    $this->switchToWorkspace($this->workspaces['open']->id());
    $nodeOpen = $this->createNode(['title' => $this->randomString(), 'status' => TRUE]);
    $this->switchToLive();

    // Create a published workspace with a node inside it.
    $this->workspaces['closed'] = Workspace::create(['id' => 'closed', 'label' => 'Closed']);
    $this->workspaces['closed']->save();
    $this->switchToWorkspace($this->workspaces['closed']->id());
    $nodeClosed = $this->createNode(['title' => $this->randomString(), 'status' => TRUE]);
    $this->switchToLive();
    $this->workspaces['closed']->publish();

    // Delete both workspaces and verify that the node created inside the
    // published workspace was not purged.
    $this->workspaces['open']->delete();
    $this->workspaces['closed']->delete();

    $storage = $this->entityTypeManager->getStorage('node');
    $node = $storage->loadUnchanged($nodeOpen->id());
    $this->assertNull($node);

    $node = $storage->loadUnchanged($nodeClosed->id());
    $this->assertEquals($node->uuid(), $nodeClosed->uuid());
  }

}
