<?php

namespace Drupal\wse_config;

use Drupal\workspaces\Event\WorkspaceSwitchEvent;
use Drupal\workspaces\WorkspaceManagerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Resolves and provides the active workspace ID.
 *
 * This service:
 * - Triggers early workspace negotiation before language negotiation.
 * - Stores the active workspace ID for config/cache access.
 * - Updates the ID when workspace switches occur.
 */
class WorkspaceIdResolver implements EventSubscriberInterface {

  /**
   * The currently active workspace ID.
   */
  protected ?string $activeWorkspaceId = NULL;

  public function __construct(
    protected readonly WorkspaceManagerInterface $workspaceManager,
  ) {}

  /**
   * Triggers workspace negotiation early in the request.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onKernelRequest(RequestEvent $event): void {
    // Only act on the main request.
    if (!$event->isMainRequest()) {
      return;
    }

    // Trigger full workspace negotiation (entity load + access check).
    $workspace = $this->workspaceManager->getActiveWorkspace();
    $this->activeWorkspaceId = $workspace?->id();
  }

  /**
   * Updates the workspace ID when workspace switches occur.
   *
   * @param \Drupal\workspaces\Event\WorkspaceSwitchEvent $event
   *   The workspace switch event.
   */
  public function onWorkspaceSwitch(WorkspaceSwitchEvent $event): void {
    $this->activeWorkspaceId = $event->getWorkspace()?->id();
  }

  /**
   * Gets the active workspace ID.
   *
   * @return string|null
   *   The active workspace ID, or NULL if not in a workspace.
   */
  public function getActiveWorkspaceId(): ?string {
    return $this->activeWorkspaceId;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run after authentication (300) but before language negotiation (255).
    $events[KernelEvents::REQUEST][] = ['onKernelRequest', 260];
    $events[WorkspaceSwitchEvent::class][] = ['onWorkspaceSwitch'];

    return $events;
  }

}
