<?php

declare(strict_types=1);

namespace Drupal\wse_config_test;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ConfigFactoryOverrideInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\wse_config\WorkspaceIdResolver;
use Symfony\Component\DependencyInjection\Attribute\AutowireDecorated;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Decorator that records config.factory get() calls before workspace is active.
 */
class ConfigFactoryRecorder implements ConfigFactoryInterface, EventSubscriberInterface {

  public function __construct(
    #[AutowireDecorated]
    protected ConfigFactoryInterface $inner,
    protected WorkspaceIdResolver $workspaceIdResolver,
    protected KeyValueFactoryInterface $keyValueFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function get($name) {
    $this->trackIfPreWorkspace($name);
    return $this->inner->get($name);
  }

  /**
   * {@inheritdoc}
   */
  public function getEditable($name) {
    $this->trackIfPreWorkspace($name);
    return $this->inner->getEditable($name);
  }

  /**
   * {@inheritdoc}
   */
  public function loadMultiple(array $names) {
    foreach ($names as $name) {
      $this->trackIfPreWorkspace($name);
    }
    return $this->inner->loadMultiple($names);
  }

  /**
   * {@inheritdoc}
   */
  public function reset($name = NULL) {
    return $this->inner->reset($name);
  }

  /**
   * {@inheritdoc}
   */
  public function rename($old_name, $new_name) {
    return $this->inner->rename($old_name, $new_name);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheKeys() {
    return $this->inner->getCacheKeys();
  }

  /**
   * {@inheritdoc}
   */
  public function clearStaticCache() {
    return $this->inner->clearStaticCache();
  }

  /**
   * {@inheritdoc}
   */
  public function listAll($prefix = '') {
    return $this->inner->listAll($prefix);
  }

  /**
   * {@inheritdoc}
   */
  public function addOverride(ConfigFactoryOverrideInterface $config_factory_override) {
    $this->inner->addOverride($config_factory_override);
  }

  /**
   * Tracks config name if accessed before workspace is available.
   */
  protected function trackIfPreWorkspace(string $name): void {
    if ($this->workspaceIdResolver->getActiveWorkspaceId() === NULL) {
      $tracked = $this->keyValueFactory->get('wse_config_test')->get('pre_workspace_configs', []);
      $tracked[$name] = TRUE;
      $this->keyValueFactory->get('wse_config_test')->set('pre_workspace_configs', $tracked);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [];
  }

}
