<?php

declare(strict_types=1);

namespace Drupal\Tests\wse_config\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\wse\Functional\WseTestUtilities;
use Drupal\wse_config\WorkspaceIdResolver;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests config objects accessed before a workspace is negotiated.
 */
#[CoversClass(WorkspaceIdResolver::class)]
#[Group('wse_config')]
#[RunTestsInSeparateProcesses]
class PreWorkspaceNegotiationConfigTest extends BrowserTestBase {

  use WseTestUtilities;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'workspaces',
    'wse_config',
    'wse_config_test',
  ];

  /**
   * Tests which config objects are accessed before a workspace is available.
   */
  public function testPreWorkspaceConfigAccess(): void {
    $role_id = $this->createAdminRole();
    $admin = $this->drupalCreateUser(values: ['roles' => [$role_id]]);
    $this->drupalLogin($admin);

    $this->setupWorkspaceSwitcherBlock();
    $this->wseCreateAndActivateWorkspaceThroughUi('Test', 'test');

    // Clear any previous tracking data.
    drupal_flush_all_caches();
    \Drupal::keyValue('wse_config_test')->delete('pre_workspace_configs');

    // Make a request to trigger the full bootstrap.
    $this->drupalGet('<front>');

    // Get tracked configs from key/value store.
    $tracked = array_keys(\Drupal::keyValue('wse_config_test')->get('pre_workspace_configs', []));
    sort($tracked);

    // Expected configs accessed before workspace negotiation.
    // This list should be kept minimal and updated if legitimate changes occur.
    $expected = [
      'system.date',
      'user.role.authenticated',
      'user.role.' . $role_id,
    ];
    sort($expected);

    // Check that only the expected config objects were requested before
    // workspace negotiation.
    $this->assertSame($expected, $tracked);
  }

}
