<?php

declare(strict_types=1);

namespace Drupal\Tests\wse_config\Functional;

use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests support for simple configuration.
 */
#[Group('wse_config')]
#[RunTestsInSeparateProcesses]
class WseConfigSimpleConfigTest extends WseConfigTestBase {

  protected const array ADMIN_PERMISSIONS = [
    ...parent::ADMIN_PERMISSIONS,
    // Add "administer site configuration" to be able to change the site name.
    'administer site configuration',
  ];

  private const string INITIAL_SITE_NAME = 'Initial site name';
  private const string NEW_SITE_NAME = 'New site name';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set the initial site name.
    \Drupal::configFactory()->getEditable('system.site')
      ->set('name', self::INITIAL_SITE_NAME)
      ->save();
  }

  /**
   * Tests basic support for simple configuration.
   */
  public function testSimpleConfigSupport(): void {
    // All users see the same site name in the live workspace at the start.
    $this->assertSiteNameByUser(
      expected_anonymous_site_name: self::INITIAL_SITE_NAME,
      expected_non_workspaces_user_site_name: self::INITIAL_SITE_NAME,
      expected_workspaces_user_1_site_name: self::INITIAL_SITE_NAME,
      expected_workspaces_user_2_site_name: self::INITIAL_SITE_NAME,
    );

    // Create a workspace for Workspaces User 1 and change the site name.
    $this->drupalLogin($this->users[self::WORKSPACES_USER_1]);
    $this->wseCreateAndActivateWorkspaceThroughUi(self::WORKSPACES_USER_1, self::WORKSPACES_USER_1);
    $this->setSiteNameThroughUi(self::NEW_SITE_NAME);
    $this->assertSiteName(
      expected: self::NEW_SITE_NAME,
      username: self::WORKSPACES_USER_1,
      message: sprintf('%s sees the new site name in their workspace.', self::WORKSPACES_USER_1),
    );

    // Create a workspace for Workspaces User 2 and check the site name.
    $this->drupalLogin($this->users[self::WORKSPACES_USER_2]);
    $this->wseCreateAndActivateWorkspaceThroughUi(self::WORKSPACES_USER_2, self::WORKSPACES_USER_2);
    $this->assertSiteName(
      expected: self::INITIAL_SITE_NAME,
      username: self::WORKSPACES_USER_2,
      message: sprintf('%s still sees the initial site name in their workspace.', self::WORKSPACES_USER_2),
    );

    // @todo Workspaces user #2 can no longer modify the site name.
    // phpcs:ignore Drupal.Commenting.InlineComment.SpacingAfter
    // @see https://www.drupal.org/project/wse/issues/3564458

    // They all still see the initial site name in the live workspace.
    $this->switchToLive();
    $this->assertSiteNameByUser(
      expected_anonymous_site_name: self::INITIAL_SITE_NAME,
      expected_non_workspaces_user_site_name: self::INITIAL_SITE_NAME,
      expected_workspaces_user_1_site_name: self::INITIAL_SITE_NAME,
      expected_workspaces_user_2_site_name: self::INITIAL_SITE_NAME,
    );

    // Workspaces user #1 publishes their workspace.
    $this->drupalLogin($this->users[self::WORKSPACES_USER_1]);
    $this->publishWorkspaceThroughUi(self::WORKSPACES_USER_1, 1);

    // They all see the new site name in the live workspace.
    $this->assertSiteNameByUser(
      expected_anonymous_site_name: self::NEW_SITE_NAME,
      expected_non_workspaces_user_site_name: self::NEW_SITE_NAME,
      expected_workspaces_user_1_site_name: self::NEW_SITE_NAME,
      expected_workspaces_user_2_site_name: self::NEW_SITE_NAME,
    );

    // @todo Workspaces user #2 can modify the site name now.
  }

  /**
   * Sets the site name through the UI.
   *
   * @param string $site_name
   *   The name to set the site name to.
   */
  protected function setSiteNameThroughUi($site_name): void {
    $edit = ['site_name' => $site_name];
    $this->drupalGet('admin/config/system/site-information');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Asserts the site name presented to each user.
   */
  private function assertSiteNameByUser(
    string $expected_anonymous_site_name,
    string $expected_non_workspaces_user_site_name,
    string $expected_workspaces_user_1_site_name,
    string $expected_workspaces_user_2_site_name,
  ): void {
    $this->assertSiteName($expected_anonymous_site_name, self::ANONYMOUS_SESSION);
    $this->assertSiteName($expected_non_workspaces_user_site_name, self::NON_WORKSPACES_USER);
    $this->assertSiteName($expected_workspaces_user_1_site_name, self::WORKSPACES_USER_1);
    $this->assertSiteName($expected_workspaces_user_2_site_name, self::WORKSPACES_USER_2);
  }

  /**
   * Asserts the site name presented to a given user.
   */
  private function assertSiteName(string $expected, string $username, ?string $message = NULL): void {
    $this->accountSwitcher->switchTo($this->users[$username]);

    // Get the site name from the front page title.
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);
    $actual = $this->getSession()
      ->getPage()
      ->find('css', 'title')
      ->getText();

    self::assertStringEndsWith(
      ' | ' . $expected,
      $actual,
      $message ?? sprintf('Got the correct site name for %s.', $username),
    );

    $this->accountSwitcher->switchBack();
  }

}
