<?php

declare(strict_types=1);

namespace Drupal\Tests\wse_config\Functional;

use Drupal\Core\Url;
use Drupal\Tests\UiHelperTrait;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\wse\Functional\WseTestUtilities;
use Drupal\user\Entity\Role;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests publishing and reverting configuration changes.
 */
#[Group('wse_config')]
#[RunTestsInSeparateProcesses]
class WseConfigTest extends WseConfigTestBase {

  use UiHelperTrait;
  use WseTestUtilities;
  use ContentTypeCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->drupalCreateContentType(['type' => 'article', 'label' => 'Article']);

    // Grant additional permissions to the test user.
    $role = Role::load($this->users[self::WORKSPACES_USER_1]->getRoles(TRUE)[0]);
    $additional_permissions = [
      'administer blocks',
      'administer nodes',
      'create article content',
    ];
    $this->grantPermissions($role, $additional_permissions);
    $this->drupalLogin($this->users[self::WORKSPACES_USER_1]);
  }

  /**
   * Test callback.
   */
  public function testWseConfig(): void {
    $block = $this->placeBlock('system_powered_by_block', [
      'label' => 'A block label',
      'label_display' => 'visible',
    ]);
    // Create a node.
    $node = $this->createNodeThroughUi('Test node', 'article');

    $this->assertSession()->pageTextContains('Test node');
    $this->assertSession()->pageTextContains('A block label');

    $workspace = $this->wseCreateAndActivateWorkspaceThroughUi('Source', 'source');

    $this->drupalGet($block->toUrl('edit-form'));
    $this->submitForm(['edit-settings-label' => 'Edited block label'], 'Save block');
    $this->assertSession()->pageTextContains('The block configuration has been saved.');

    $this->drupalGet($node->toUrl('edit-form'));
    $this->submitForm(['title[0][value]' => 'Test edited node'], 'Save');

    $this->assertSession()->pageTextContains('Test edited node');
    $this->assertSession()->pageTextContains('Edited block label');

    $this->assertSession()->buttonExists('Switch to Live')->press();

    $this->assertSession()->pageTextContains('Test node');
    $this->assertSession()->pageTextContains('A block label');

    $this->assertSession()->selectExists('workspace_id')->selectOption($workspace->id());
    $this->assertSession()->buttonExists('Activate')->press();

    $this->assertSession()->pageTextContains('Test edited node');
    $this->assertSession()->pageTextContains('Edited block label');

    $this->drupalGet(Url::fromRoute('entity.workspace.publish_form', ['workspace' => $workspace->id()]));
    $this->assertSession()->pageTextContains('1 content item');
    $this->assertSession()->pageTextContains('1 workspace config');
    $this->assertSession()->buttonExists('Publish 2 items to Live')->press();

    $this->drupalGet($node->toUrl());
    $this->assertSession()->pageTextContains('Test edited node');
    $this->assertSession()->pageTextContains('Edited block label');
    // Ensure we are in the live workspace.
    $this->assertSession()->pageTextContains('Current workspace None');

    // Try to revert the workspace.
    $this->drupalGet(Url::fromRoute('entity.workspace.revert_form', ['workspace' => $workspace->id()]));
    $this->assertSession()->buttonExists('Revert')->press();
    $this->drupalGet($node->toUrl());
    $this->assertSession()->pageTextContains('Test node');
    $this->assertSession()->pageTextContains('A block label');
  }

}
