<?php

namespace Drupal\wse\Form;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\WorkspaceSafeFormInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\workspaces\WorkspaceInterface;
use Drupal\workspaces\WorkspaceTrackerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the form for moving an entity to a different workspace.
 */
class MoveEntityForm extends ConfirmFormBase implements WorkspaceSafeFormInterface, ContainerInjectionInterface {

  use EntityDependencyTrait;

  /**
   * The entity that will be moved.
   *
   * @var \Drupal\Core\Entity\RevisionableInterface
   */
  protected RevisionableInterface $entity;

  /**
   * The workspace where the entity will be moved from.
   *
   * @var \Drupal\workspaces\WorkspaceInterface
   */
  protected WorkspaceInterface $sourceWorkspace;

  public function __construct(
    protected WorkspaceTrackerInterface $workspaceTracker,
    protected SelectionPluginManagerInterface $selectionHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('workspaces.tracker'),
      $container->get('plugin.manager.entity_reference_selection'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'workspace_move_entity_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?RouteMatchInterface $route_match = NULL, ?WorkspaceInterface $source_workspace = NULL) {
    $this->entity = $route_match->getParameter($route_match->getParameter('entity_type_id'));
    $this->sourceWorkspace = $source_workspace;
    $form = parent::buildForm($form, $form_state);

    $workspace_selection = $this->selectionHandler->getInstance(['target_type' => 'workspace']);
    $options = $workspace_selection->getReferenceableEntities();
    $options = reset($options);

    // Remove the source workspace from the list of target workspaces.
    unset($options[$source_workspace->id()]);

    // Filter out closed workspaces.
    $workspaces = \Drupal::entityTypeManager()->getStorage('workspace')->loadMultiple(array_keys($options));
    foreach (array_keys($options) as $workspace_id) {
      if (wse_workspace_get_status($workspaces[$workspace_id]) !== WSE_STATUS_OPEN) {
        unset($options[$workspace_id]);
      }
    }

    $form['target_workspace'] = [
      '#type' => 'select',
      '#options' => $options,
      '#title' => $this->t('Target workspace'),
      '#description' => $this->t('The workspace where this entity should be moved.'),
      '#required' => TRUE,
    ];

    $this->addDependenciesFormElements($form, $form_state, $this->entity, $this->sourceWorkspace);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $target_workspace_id = $form_state->getValue('target_workspace');
    $source_workspace_id = $this->sourceWorkspace->id();

    // Build list of entities to move.
    $affected_entity_ids[$this->entity->getEntityTypeId()][$this->entity->id()] = TRUE;
    $affected_entity_ids = array_replace_recursive($affected_entity_ids, $this->getSelectedDependencies($form_state));

    try {
      foreach ($affected_entity_ids as $entity_type_id => $entity_ids) {
        $this->workspaceTracker->moveTrackedEntities(
          $source_workspace_id,
          $target_workspace_id,
          $entity_type_id,
          array_keys($entity_ids)
        );
      }
      $this->messenger()->addStatus($this->t('Successful operation.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Operation failed. All errors have been logged.'));
      $this->getLogger('workspaces')->error($e->getMessage());
    }

    $form_state->setRedirectUrl($this->sourceWorkspace->toUrl());
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Move entity to another workspace');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('Move entity to another workspace.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return $this->sourceWorkspace->toUrl();
  }

}
