<?php

declare(strict_types=1);

namespace Drupal\wse\Hook;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\Order;
use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\workspaces\WorkspaceManagerInterface;

/**
 * Form hooks for the wse module.
 *
 * @internal
 */
class WseFormHooks {

  /**
   * Collects altered form ids to pass them down to the form submit dialog JS.
   *
   * @var array
   */
  protected static array $alteredFormIds = [];

  public function __construct(
    protected readonly WorkspaceManagerInterface $workspaceManager,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter', order: Order::Last)]
  public function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // No alterations are needed if we're not in a workspace context.
    if (!$this->workspaceManager->hasActiveWorkspace() || $form_state->get('workspace_safe')) {
      return;
    }

    // @todo Fix in core.
    $safe_form_ids = [
      'node_revision_revert_confirm',
      'node_revision_delete_confirm',
    ];
    // Allow a few forms that we know are safe to submit.
    if (in_array($form_id, $safe_form_ids, TRUE)) {
      $form_state->set('workspace_safe', TRUE);
      return;
    }

    $ignored_forms = $this->configFactory->get('wse.settings')->get('safe_forms') ?? [];
    if (!in_array($form_id, $ignored_forms)) {
      // The alterations here make core's validation obsolete, so remove it.
      $this->removeWorkspaceValidation($form);

      // Add a form element that will be used by WseWorkspaceManager to disable
      // any active workspace for the duration of the form submission request.
      $form['wse_bypass_workspace'] = [
        '#type' => 'hidden',
        '#value' => 1,
        '#name' => 'wse_bypass_workspace',
        // Form processing and validation require this value. Ensure the
        // submitted form value appears literally, regardless of custom #tree
        // and #parents being set elsewhere.
        '#parents' => ['wse_bypass_workspace'],
      ];

      // The confirm submission dialog is implemented on the client side.
      $form['#attached']['library'][] = 'wse/form-submit-dialog';
      static::$alteredFormIds[] = Html::getId($form_id);
      $form['#attached']['drupalSettings']['wseSubmitDialog']['formSelectors'] = static::$alteredFormIds;
    }
  }

  /**
   * Removes workspaces core's validation handler recursively on each element.
   *
   * @param array &$element
   *   An associative array containing the structure of the form.
   */
  protected function removeWorkspaceValidation(array &$element): void {
    // Recurse through all children and add our validation handler if needed.
    foreach (Element::children($element) as $key) {
      if (isset($element[$key]) && $element[$key]) {
        $this->removeWorkspaceValidation($element[$key]);
      }
    }

    if (isset($element['#validate'])) {
      foreach ($element['#validate'] as $key => $validation_callback) {
        if (is_array($validation_callback) && in_array('validateDefaultWorkspace', $validation_callback)) {
          unset($element['#validate'][$key]);
          return;
        }
      }
    }
  }

  /**
   * Implements hook_form_FORM_ID_alter() for 'workspace_publish_form'.
   */
  #[Hook('form_workspace_publish_form_alter')]
  public function formWorkspacePublishFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    $form['clone_on_publish'] = [
      '#type' => 'checkbox',
      '#title' => new TranslatableMarkup('Clone workspace details into a new draft workspace'),
      '#default_value' => $this->configFactory->get('wse.settings')->get('clone_on_publish'),
    ];

    $form['#validate'][] = [self::class, 'workspacePublishFormValidate'];
    $form['#submit'][] = [self::class, 'workspacePublishFormSubmit'];
  }

  /**
   * Validation callback for the workspace publish form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function workspacePublishFormValidate(array $form, FormStateInterface $form_state): void {
    /** @var \Drupal\workspaces\WorkspaceInterface $workspace */
    $workspace = $form_state->getFormObject()->getWorkspace();
    $workspace->_clone_on_publish = $form_state->getValue('clone_on_publish');
  }

  /**
   * Submit callback for the workspace publish form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function workspacePublishFormSubmit(array $form, FormStateInterface $form_state): void {
    // @todo Fix this upstream.
    $form_state->setRedirect('entity.workspace.collection');
  }

}
