<?php

namespace Drupal\wse;

use Drupal\workspaces\WorkspaceInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Symfony\Component\DependencyInjection\Attribute\AutowireDecorated;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a workspace manager service override.
 */
class WseWorkspaceManager implements WorkspaceManagerInterface {

  public function __construct(
    #[AutowireDecorated]
    protected WorkspaceManagerInterface $inner,
    protected RequestStack $requestStack,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function hasActiveWorkspace() {
    $request = $this->requestStack->getCurrentRequest();
    if ($request && $request->request->has('wse_bypass_workspace')) {
      return FALSE;
    }

    return $this->getActiveWorkspace() !== NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getActiveWorkspace() {
    $request = $this->requestStack->getCurrentRequest();

    // When there is no request in the stack, return early and allow following
    // calls to try and determine the active workspace.
    if (!$request) {
      return NULL;
    }

    if ($request && $request->request->has('wse_bypass_workspace')) {
      return NULL;
    }

    // Don't allow closed workspaces to be activated.
    $negotiated_workspace = $this->inner->getActiveWorkspace();
    if ($negotiated_workspace instanceof WorkspaceInterface && wse_workspace_get_status($negotiated_workspace) == WSE_STATUS_CLOSED) {
      return NULL;
    }

    return $negotiated_workspace;
  }

  /**
   * {@inheritdoc}
   */
  public function setActiveWorkspace(WorkspaceInterface $workspace, bool $persist = TRUE) {
    return $this->inner->setActiveWorkspace($workspace, $persist);
  }

  /**
   * {@inheritdoc}
   */
  public function switchToLive() {
    return $this->inner->switchToLive();
  }

  /**
   * {@inheritdoc}
   */
  public function executeInWorkspace($workspace_id, callable $function) {
    return $this->inner->executeInWorkspace($workspace_id, $function);
  }

  /**
   * {@inheritdoc}
   */
  public function executeOutsideWorkspace(callable $function) {
    return $this->inner->executeOutsideWorkspace($function);
  }

  /**
   * {@inheritdoc}
   */
  public function purgeDeletedWorkspacesBatch(): void {
    $this->inner->purgeDeletedWorkspacesBatch();
  }

}
