<?php

declare(strict_types=1);

namespace Drupal\Tests\wse\Functional;

use Drupal\Core\StringTranslation\PluralTranslatableMarkup;
use Drupal\Core\Url;
use Drupal\Tests\workspaces\Functional\WorkspaceTestUtilities;
use Drupal\workspaces\WorkspaceInterface;

/**
 * Utility methods for use in BrowserTestBase tests.
 *
 * This trait will not work if not used in a child of BrowserTestBase.
 */
trait WseTestUtilities {

  use WorkspaceTestUtilities;

  /**
   * Creates and activates a new Workspace through the UI.
   *
   * @param string $label
   *   The label of the workspace to create.
   * @param string $id
   *   The ID of the workspace to create.
   * @param string $parent
   *   (optional) The ID of the parent workspace. Defaults to '_none'.
   *
   * @return \Drupal\workspaces\WorkspaceInterface
   *   The workspace that was just created.
   */
  protected function wseCreateAndActivateWorkspaceThroughUi(string $label, string $id, string $parent = '_none'): WorkspaceInterface {
    $this->drupalGet('/admin/config/workflow/workspaces/add');
    $this->submitForm([
      'id' => $id,
      'label' => $label,
      'parent' => $parent,
    ], 'Save and switch');

    $workspace = $this->getOneEntityByLabel('workspace', $label);

    $this->getSession()->getPage()->hasContent("$label ({$workspace->id()})");

    return $workspace;
  }

  /**
   * Creates a new Workspace through the UI.
   *
   * @param string $label
   *   The label of the workspace to create.
   * @param string $id
   *   The ID of the workspace to create.
   * @param string $parent
   *   (optional) The ID of the parent workspace. Defaults to '_none'.
   *
   * @return \Drupal\workspaces\WorkspaceInterface
   *   The workspace that was just created.
   */
  protected function wseCreateWorkspaceThroughUi($label, $id, $parent = '_none'): WorkspaceInterface {
    $this->drupalGet('/admin/config/workflow/workspaces/add');
    $this->submitForm([
      'id' => $id,
      'label' => $label,
      'parent' => $parent,
    ], 'Save');

    $workspace = $this->getOneEntityByLabel('workspace', $label);

    $this->getSession()->getPage()->hasContent("$label ({$workspace->id()})");

    return $workspace;
  }

  /**
   * Publishes a workspace through the UI.
   *
   * @param string $label
   *   The label of the workspace to publish.
   * @param int $item_count
   *   The number of changes that will be published.
   */
  protected function publishWorkspaceThroughUi(string $label, int $item_count): void {
    $workspace = $this->getOneEntityByLabel('workspace', $label);

    $this->drupalGet(Url::fromRoute('entity.workspace.publish_form', ['workspace' => $workspace->id()]));
    $this->assertSession()->statusCodeEquals(200);

    $button_label = new PluralTranslatableMarkup($item_count, 'Publish @count item to Live', 'Publish @count items to Live', ['@count' => $item_count]);
    $this->getSession()->getPage()->pressButton((string) $button_label);
    $this->assertSession()->pageTextContains('Successful publication.');
  }

}
