<?php

namespace Drupal\x_frame_options_configuration\EventSubscriber;

use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Subscribing an event.
 */
class XframeSubscriber implements EventSubscriberInterface {

  /**
   * Drupal's settings manager.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * XframeSubscriber constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->config = $config_factory->get('x_frame_options_configuration.settings');
  }

  /**
   * Executes actions on the respose event.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   Filter Response Event object.
   */
  public function onKernelResponse(ResponseEvent $event) {
    // Add the x-frame-options response header with the configured directive.
    $directive = $this->config->get('x_frame_options_configuration.directive', 0);
    $allow_from_uri = Html::escape($this->config->get('x_frame_options_configuration.allow-from-uri', ''));
    $x_frame_options = Html::escape($directive) . (($directive == 'ALLOW-FROM') ? " " . UrlHelper::stripDangerousProtocols($allow_from_uri) : '');

    $response = $event->getResponse();
    // If option selected is ALLOW-ALL, removes header.
    if ($x_frame_options == 'ALLOW-ALL') {
      $response->headers->remove('X-Frame-Options');
    }
    else {
      $response->headers->set('X-Frame-Options', $x_frame_options);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Adds the event in the list of KernelEvents::RESPONSE with priority -20.
    $events[KernelEvents::RESPONSE][] = ['onKernelResponse', -20];
    return $events;
  }

}
