<?php

namespace Drupal\xls_serialization_extras\Encoder;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\xls_serialization\Encoder\Xls as OriginalXls;
use PhpOffice\PhpSpreadsheet\Cell\Cell;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

/**
 * Adds XLS encoder support for the Serialization API.
 */
class Xls extends OriginalXls {

  /**
   * Constructs an XLS encoder.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param string $xls_format
   *   The XLS format to use.
   */
  public function __construct($config_factory, protected ModuleHandlerInterface $moduleHandler, $xls_format = 'Xlsx') {
    parent::__construct($config_factory, $xls_format);
  }

  /**
   * {@inheritdoc}
   */
  protected function setStyles(Worksheet $sheet, array $data, array $context) {
    parent::setStyles($sheet, $data, $context);
    // Style columns using the Excel field formatters.
    if (isset($context['views_style_plugin']->view->field)) {
      $fields = $context['views_style_plugin']->view->field;
      $this->styleColumnsUsingFieldFormatters($sheet, $data, $fields);
    }
    // Let modules alter the active Excel sheet.
    $this->moduleHandler->alter('xls_serialization_extras_worksheet', $sheet, $context);
  }

  /**
   * Apply styling to the column headers using the field formatters.
   *
   * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
   *   The worksheet to work on.
   * @param array $data
   *   The content data array.
   * @param \Drupal\views\Plugin\views\field\FieldPluginBase[] $fields
   *   The view fields.
   */
  protected function styleColumnsUsingFieldFormatters(Worksheet $sheet, array $data, array $fields): void {
    $column = 1;
    foreach ($fields as $field) {
      $settings = $field->options['settings'] ?? [];
      if (isset($settings['excel_formatter_class'])) {
        // Define column collapse status.
        if ($settings['excel_collapse'] ?? FALSE) {
          $column_string = Coordinate::stringFromColumnIndex($column);
          $column_dimension = $sheet->getColumnDimension($column_string);
          $column_dimension->setCollapsed(TRUE);
          $column_dimension->setVisible(FALSE);
        }
        // Define column alignment including header row.
        if (!empty($settings['excel_horizontal_alignment'])) {
          $style = $sheet->getStyle([$column, 1, $column, count($data) + 1]);
          $alignment = $style->getAlignment();
          switch ($settings['excel_horizontal_alignment']) {
            case 'left':
              $alignment->setHorizontal(Alignment::HORIZONTAL_LEFT);
              break;

            case 'center':
              $alignment->setHorizontal(Alignment::HORIZONTAL_CENTER);
              break;

            case 'right':
              $alignment->setHorizontal(Alignment::HORIZONTAL_RIGHT);
              break;
          }
        }
        // Define column text styling excluding header row.
        if (isset($settings['excel_bold']) || isset($settings['excel_italic'])) {
          $style = $sheet->getStyle([$column, 2, $column, count($data) + 1]);
          if (isset($settings['excel_bold'])) {
            $style->getFont()->setBold((bool) $settings['excel_bold']);
          }
          if (isset($settings['excel_italic'])) {
            $style->getFont()->setItalic((bool) $settings['excel_italic']);
          }
        }
      }
      $column++;
    }
  }

  /**
   * Set sheet data.
   *
   * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
   *   The worksheet to put the data in.
   * @param array $data
   *   The data to be put in the worksheet.
   */
  protected function setData(Worksheet $sheet, array $data) {
    // Since headers have been added, rows start at 2.
    $rowCount = 2;
    foreach ($data as $row) {
      $column = 1;
      // Required.
      // @see https://www.drupal.org/project/xls_serialization/issues/3362321
      $row = (array) $row;
      foreach ($row as $value) {
        if (is_array($value)) {
          $this->processCellDataArray($sheet, $value, $column, $rowCount);
        }
        else {
          $this->setValueDefault($sheet, $value, $column, $rowCount);
        }
        $column++;
      }
      $rowCount++;
    }
  }

  /**
   * Processes the cell data and applies styles or images if needed.
   *
   * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
   *   The worksheet to process the cell data in.
   * @param array $data
   *   The data array to be processed, which includes type and attributes.
   * @param int $column
   *   The column index where the value should be placed.
   * @param int $row
   *   The row index where the value should be placed.
   */
  protected function processCellDataArray(Worksheet $sheet, array $data, int $column, int $row): void {
    if (isset($data['type'])) {
      switch ($data['type']) {
        case 'string':
          $formattedValue = $this->formatValue($data['value']);
          $enforce_string = $data['attributes']['enforce_string'] ?? FALSE;
          if ($enforce_string || $this->isFormulaString($formattedValue)) {
            $sheet->setCellValueExplicit([$column, $row], $formattedValue, DataType::TYPE_STRING);
          }
          else {
            $sheet->setCellValue([$column, $row], $formattedValue);
          }
          break;

        case 'numeric':
          $value = $data['value'];
          if (is_numeric($value)) {
            if (strlen((string) $value) > 11) {
              $big_number_as_string = $data['attributes']['big_number_as_string'] ?? FALSE;
              if ($big_number_as_string) {
                // Workaround for big number to avoid scientific formatting.
                // Ex: number 111111111111 won't be converted to 1,11111E+11.
                // @see https://www.drupal.org/project/xls_serialization/issues/3056686
                $sheet->setCellValueExplicit([$column, $row], (string) $value, DataType::TYPE_STRING);
              }
              else {
                $sheet->setCellValueExplicit([$column, $row], $value, DataType::TYPE_NUMERIC);
              }
            }
            else {
              $sheet->setCellValueExplicit([$column, $row], $value, DataType::TYPE_NUMERIC);
            }
          }
          else {
            $this->setValueDefault($sheet, $value, $column, $row);
          }
          break;

        case 'datetime':
        case 'timestamp':
          $attributes = $data['attributes'];
          if (isset($attributes['datetime'])) {
            $date_time = $attributes['datetime'];
            $cell = $sheet->getCell([$column, $row]);
            $cell->setValueExplicit($date_time, DataType::TYPE_ISO_DATE);
            if (isset($attributes['pattern'])) {
              $pattern = $attributes['pattern'];
            }
            else {
              $pattern = NumberFormat::FORMAT_DATE_DATETIME;
            }
            $cell->getStyle()->getNumberFormat()->setFormatCode($pattern);
          }
          else {
            $this->setValueDefault($sheet, $data['value'], $column, $row);
          }
          break;

        case 'image':
          $this->processImageData($sheet, $data, $column, $row);
          break;

        case 'formula':
          $sheet->setCellValueExplicit([$column, $row], $data['value'], DataType::TYPE_FORMULA);
          break;

        default:
          $this->setValueDefault($sheet, $data['value'], $column, $row);
          break;
      }
      if (!empty($data['attributes'])) {
        $cell = $sheet->getCell([$column, $row]);
        $this->styleCell($cell, $data['attributes']);
      }
    }
  }

  /**
   * Default data rendering in the cell.
   *
   * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
   *   The worksheet to set the data in.
   * @param mixed $value
   *   The value to be set.
   * @param int $column
   *   The column index where the value should be placed.
   * @param int $row
   *   The row index where the value should be placed.
   */
  protected function setValueDefault(Worksheet $sheet, mixed $value, int $column, int $row): void {
    $formattedValue = $this->formatValue($value);
    if ($this->isFormulaString($formattedValue)) {
      $sheet->setCellValueExplicit([$column, $row], $formattedValue, DataType::TYPE_STRING);
    }
    else {
      $sheet->setCellValue([$column, $row], $formattedValue);
    }
  }

  /**
   * Processes the cell image data.
   *
   * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
   *   The worksheet to process the cell data in.
   * @param array $data
   *   The data array to be processed, which includes type and attributes.
   * @param int $column
   *   The column index where the value should be placed.
   * @param int $row
   *   The row index where the value should be placed.
   */
  protected function processImageData(Worksheet $sheet, array $data, int $column, int $row) {
    try {
      // PNG, JPEG, GIF supported.
      $drawing = new Drawing();
      $absolute_path = $data['value'];
      $drawing->setPath($absolute_path);
      // Set height in pixels.
      $desired_height = 80;
      $drawing->setHeight($desired_height);
      // Cell where the image's top-left corner will be anchored.
      $column_letter = Coordinate::stringFromColumnIndex($column);
      $drawing->setCoordinates($column_letter . $row);
      $drawing->setWorksheet($sheet);
      $image_size = getimagesize($absolute_path);
      $image_ratio = $image_size[0] / $image_size[1];
      $desired_width = $desired_height * $image_ratio;
      $row_dimension = $sheet->getRowDimension($row);
      $row_height = $row_dimension->getRowHeight();
      if ($row_height < $desired_height * 0.75) {
        $row_dimension->setRowHeight($desired_height * 0.75);
      }
      $col_dimension = $sheet->getColumnDimension($column_letter);
      $col_width = $col_dimension->getWidth();
      if ($col_width < $desired_width / 7) {
        $col_dimension->setWidth($desired_width / 7);
      }
    }
    catch (\Exception $e) {
      // @todo Handle exception if needed.
    }
  }

  /**
   * Styles a cell using the provided attributes.
   *
   * @param \PhpOffice\PhpSpreadsheet\Cell\Cell $cell
   *   The cell to style.
   * @param array $attributes
   *   The settings to apply to the cell, such as alignment or font style.
   */
  protected function styleCell(Cell $cell, array $attributes): void {
    $style = $cell->getStyle();
    if (isset($attributes['bold'])) {
      $style->getFont()->setBold((bool) $attributes['bold']);
    }
    if (isset($attributes['italic'])) {
      $style->getFont()->setItalic((bool) $attributes['italic']);
    }
    if (!empty($attributes['horizontal_alignment'])) {
      $alignment = $style->getAlignment();
      switch ($attributes['horizontal_alignment']) {
        case 'left':
          $alignment->setHorizontal(Alignment::HORIZONTAL_LEFT);
          break;

        case 'center':
          $alignment->setHorizontal(Alignment::HORIZONTAL_CENTER);
          break;

        case 'right':
          $alignment->setHorizontal(Alignment::HORIZONTAL_RIGHT);
          break;
      }
    }
  }

  /**
   * Set width of all columns with data in them in sheet to AutoSize.
   *
   * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
   *   The worksheet to set the column width to AutoSize for.
   */
  protected function setColumnsAutoSize(Worksheet $sheet) {
    foreach ($sheet->getColumnIterator() as $column) {
      $column_index = $column->getColumnIndex();
      $col_dimension = $sheet->getColumnDimension($column_index);
      // Do not reset the width if it has already been modified.
      if (!($col_dimension->getWidth() > 0)) {
        $col_dimension->setAutoSize(TRUE);
      }
    }
  }

  /**
   * Sets height of all rows to automatic and enables text wrapping.
   *
   * @param \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet
   *   The worksheet to set the column width to AutoSize for.
   */
  protected function setRowsAutoHeight(Worksheet $sheet) {
    foreach ($sheet->getRowDimensions() as $row_dimension) {
      // Do not reset the height if it has already been modified.
      if (!($row_dimension->getRowHeight() > 0)) {
        $row_dimension->setRowHeight(-1);
      }
    }
    $sheet->getStyle('A1:' . $sheet->getHighestColumn() . $sheet->getHighestRow())->getAlignment()->setWrapText(TRUE);
  }

  /**
   * Checks if the formatted value is a formula string.
   *
   * @param mixed $value
   *   The formatted value to check.
   *
   * @return bool
   *   True if the formatted value is a formula string, false otherwise.
   */
  protected function isFormulaString(mixed $value): bool {
    return is_string($value) && strlen($value) > 1 && $value[0] === '=';
  }

}
