<?php

namespace Drupal\xls_serialization_extras\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\ResultRow;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\xls_serialization_extras\Render\ExcelDataMarkup;

/**
 * Provides a Views field that outputs an Excel formula.
 *
 * @ingroup views_field_handlers
 */
#[ViewsField("excel_formula")]
class ExcelFormulaField extends FieldPluginBase {

  /**
   * {@inheritdoc}
   */
  public function usesGroupBy() {
    return FALSE;
  }

  /**
   * Define options for the field.
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['formula'] = ['default' => '=A{{ row }}+B{{ row }}'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    $form['formula'] = [
      '#title' => $this->t('Formula'),
      '#type' => 'textfield',
      '#description' => $this->t('Use Excel-style formulas. Use {{ row }} as a placeholder for the current row.'),
      '#default_value' => $this->options['formula'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Prevent Views from adding this field to the SQL query.
    $this->field_alias = $this->realField = NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ResultRow $values, $field = NULL) {
    return $this->options['formula'];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    // No need to sanitize the formula as it is not user input.
    // The formula is defined in the options and is safe to use.
    return $this->getValue($values);
  }

  /**
   * {@inheritdoc}
   */
  public function renderText($alter) {
    $value = (string) $this->last_render;
    $tokens = $this->getRenderTokens($alter);
    $value = strtr($value, $tokens);
    $this->last_render_text = $value;
    return ExcelDataMarkup::create('formula', $value);
  }

  /**
   * {@inheritdoc}
   */
  protected function addSelfTokens(&$tokens, $item) {
    $tokens['{{ row }}'] = $this->view->row_index + 2;
  }

}
