<?php

namespace Drupal\xls_serialization_extras\Plugin\Field\FieldFormatter;

use Drupal\Core\Datetime\Entity\DateFormat;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\TimestampFormatter;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'excel_timestamp' formatter.
 */
#[FieldFormatter(
  id: 'excel_timestamp',
  label: new TranslatableMarkup('Excel'),
  description: new TranslatableMarkup('Allows to add styling to an Excel timestamp column.'),
  field_types: [
    'timestamp',
    'created',
    'changed',
  ]
)]
class ExcelTimestampFormatter extends TimestampFormatter {

  use ExcelFormatterTrait;

  /**
   * {@inheritdoc}
   */
  public function prepareView(array $entities_items) {
    parent::prepareView($entities_items);
    $format_name = $this->getSetting('date_format');
    if ($format_name === 'custom') {
      $date_pattern = $this->getSetting('custom_date_format');
    }
    else {
      $date_format = DateFormat::load($format_name);
      $date_pattern = $date_format ? $date_format->getPattern() : NULL;
    }
    if ($date_pattern) {
      $excel_pattern = ExcelFormatterUtils::phpToExcelDatePattern($date_pattern);
      $this->setSetting('excel_pattern', $excel_pattern);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = parent::viewElements($items, $langcode);
    $excel_pattern = $this->getSetting('excel_pattern');
    foreach ($items as $delta => $item) {
      $element = $elements[$delta];
      $attributes = $element['#attributes'];
      if ($excel_pattern) {
        $attributes['pattern'] = $excel_pattern;
      }
      $elements[$delta] = [
        '#type' => 'excel_data',
        '#data_type' => 'timestamp',
        '#value' => $element['#text'],
        '#attributes' => $attributes,
      ];
    }
    return $elements;
  }

}
