<?php

namespace Drupal\xls_serialization_extras\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\NumericUnformattedFormatter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'excel_string' formatter.
 */
#[FieldFormatter(
  id: 'excel_numeric_unformatted',
  label: new TranslatableMarkup('Excel (Unformatted)'),
  field_types: [
    'integer',
    'decimal',
    'float',
  ]
)]
class ExcelNumericUnformattedFormatter extends NumericUnformattedFormatter {

  use ExcelFormatterTrait;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return parent::defaultSettings() + static::commonSettings() + [
      'big_number_as_string' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);
    $form = $this->commonSettingsForm($form, $form_state);
    $form['big_number_as_string'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Big numbers as strings'),
      '#default_value' => $this->getSetting('big_number_as_string'),
      '#description' => $this->t('If checked, big numbers will be treated as strings to prevent scientific formatting.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $summary = $this->commonSettingsSummary($summary);
    if ($this->getSetting('big_number_as_string')) {
      $summary[] = $this->t('Big numbers are treated as strings.');
    }
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    $attributes = [];
    if ($this->getSetting('big_number_as_string')) {
      $attributes['big_number_as_string'] = TRUE;
    }

    foreach ($items as $delta => $item) {
      $elements[$delta] = [
        '#type' => 'excel_data',
        '#data_type' => 'numeric',
        '#value' => $item->value,
        '#attributes' => $attributes,
      ];
    }

    return $elements;
  }

}
