<?php

namespace Drupal\xls_serialization_extras\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\StringFormatter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'excel_string' formatter.
 */
#[FieldFormatter(
  id: 'excel_string',
  label: new TranslatableMarkup('Excel'),
  description: new TranslatableMarkup('Allows to add styling to an Excel string column.'),
  field_types: [
    'string',
    'string_long',
    'email',
    'uri',
  ]
)]
class ExcelStringFormatter extends StringFormatter {

  use ExcelFormatterTrait;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return parent::defaultSettings() + static::commonSettings() + [
      'enforce_string' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);
    $form = $this->commonSettingsForm($form, $form_state);
    $form['enforce_string'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enforce string type'),
      '#default_value' => $this->getSetting('enforce_string'),
      '#description' => $this->t('If checked, the value will be enforced as a string, preventing any numeric interpretation. This is useful for fields that may contain numeric values but should be treated as text, such as phone numbers or IDs.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $summary = $this->commonSettingsSummary($summary);
    if ($this->getSetting('enforce_string')) {
      $summary[] = $this->t('Enforce string type is enabled.');
    }
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = parent::viewElements($items, $langcode);
    foreach ($elements as $delta => $element) {
      if ($element['#type'] === 'link') {
        $url = $element['#url'];
        $url->setAbsolute(TRUE);
        $elements[$delta] = $element['#title'];
        $elements[$delta]['#attributes']['url'] = $url->toString();
      }
    }
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  protected function viewValue(FieldItemInterface $item) {
    $attributes = [];
    if ($this->getSetting('enforce_string')) {
      $attributes['enforce_string'] = TRUE;
    }
    return [
      '#type' => 'excel_data',
      '#data_type' => 'string',
      '#value' => $item->value,
      '#attributes' => $attributes,
    ];
  }

}
