<?php

namespace Drupal\xls_serialization_extras\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\ResultRow;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ViewExecutable;
use Drupal\xls_serialization_extras\Render\ExcelDataMarkup;

/**
 * Provides a Views field that outputs an Excel formula.
 *
 * @ingroup views_field_handlers
 */
#[ViewsField("excel_formula")]
class ExcelFormulaField extends FieldPluginBase implements ExcelFieldInterface {

  use ExcelFieldTrait;

  /**
   * The token processor for handling formula tokens.
   *
   * @var \Drupal\xls_serialization_extras\Plugin\views\field\FormulaTokenProcessor
   */
  protected $tokenProcessor;

  /**
   * The extracted formula tokens.
   *
   * @var array
   */
  protected $formulaTokens;

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, ?array &$options = NULL) {
    parent::init($view, $display, $options);
    $this->tokenProcessor = new FormulaTokenProcessor();
    $this->formulaTokens = $this->tokenProcessor->extractTokens($this->options['formula']);
  }

  /**
   * {@inheritdoc}
   */
  public function usesGroupBy() {
    return FALSE;
  }

  /**
   * Define options for the field.
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['formula'] = ['default' => '=A{{ row }}+B{{ row }}'];
    return $options + static::commonOptions();
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    // Remove options that are not applicable for this field type.
    unset($form['exclude']);
    // Add the formula option.
    $form['formula'] = [
      '#title' => $this->t('Formula'),
      '#type' => 'textfield',
      '#description' => $this->t('Use Excel-style formulas. Use {{ row }} as a placeholder for the current row.'),
      '#default_value' => $this->options['formula'],
    ];
    $this->commonOptionsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Prevent Views from adding this field to the SQL query.
    $this->field_alias = $this->realField = NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ResultRow $values, $field = NULL) {
    return $this->options['formula'];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    // No need to sanitize the formula as it is not user input.
    // The formula is defined in the options and is safe to use.
    return $this->getValue($values);
  }

  /**
   * {@inheritdoc}
   */
  public function renderText($alter) {
    $value = $this->options['formula'];
    if (!empty($this->formulaTokens)) {
      $replacements = [];
      $tokens = $this->getRenderTokens($alter);
      foreach ($this->formulaTokens as $replacement => $formula_token) {
        $token_value = $tokens['{{ ' . $formula_token['token'] . ' }}'] ?? '';
        if ($filter = $formula_token['filter'] ?? NULL) {
          $token_value = $this->tokenProcessor->applyFilter((string) $token_value, $filter);
        }
        $replacements[$replacement] = $token_value;
      }
      $value = strtr($value, $replacements);
    }
    $this->last_render_text = $value;
    return ExcelDataMarkup::create('formula', $value);
  }

  /**
   * {@inheritdoc}
   */
  protected function addSelfTokens(&$tokens, $item) {
    $tokens['{{ row }}'] = $this->view->row_index + 2;
  }

}
