<?php

namespace Drupal\xls_serialization_extras\Plugin\Field\FieldFormatter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\xls_serialization_extras\Plugin\ExcelStylesTrait;

/**
 * Trait for the Excel formatters.
 */
trait ExcelFormatterTrait {

  use ExcelStylesTrait;

  /**
   * Returns the common settings for the Excel formatter.
   *
   * @return array
   *   An associative array of default settings.
   */
  protected static function commonSettings() {
    return [
      'excel_bold' => FALSE,
      'excel_italic' => FALSE,
      'excel_underline' => FALSE,
      'excel_color' => '',
      'excel_horizontal_alignment' => '',
      'excel_collapse' => FALSE,
      'excel_formatter_class' => static::class,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return parent::defaultSettings() + static::commonSettings();
  }

  /**
   * Builds the default settings form for the Excel formatter.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The modified form structure.
   */
  protected function commonSettingsForm(array $form, FormStateInterface $form_state) {
    $settings = $this->getSettings();
    $this->buildStylesSettingsForm($form, $settings);
    $form['excel_formatter_class'] = [
      '#type' => 'hidden',
      '#default_value' => static::class,
    ];
    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);
    return $this->commonSettingsForm($form, $form_state);
  }

  /**
   * Provides a summary of the default settings for the Excel formatter.
   *
   * @param array $summary
   *   The existing summary array to append to.
   *
   * @return array
   *   The updated summary array with the default settings.
   */
  protected function commonSettingsSummary(array $summary) {
    if ($this->getSetting('excel_bold')) {
      $summary[] = $this->t('Excel: Bold');
    }
    if ($this->getSetting('excel_italic')) {
      $summary[] = $this->t('Excel: Italic');
    }
    if ($this->getSetting('excel_underline')) {
      $summary[] = $this->t('Excel: Underline');
    }
    if (!empty($this->getSetting('excel_color'))) {
      $summary[] = $this->t('Excel: Color: #@color', [
        '@color' => $this->getSetting('excel_color'),
      ]);
    }
    if (!empty($this->getSetting('excel_horizontal_alignment'))) {
      $summary[] = $this->t('Excel: Horizontal alignment: @alignment', [
        '@alignment' => $this->getSetting('excel_horizontal_alignment'),
      ]);
    }
    if ($this->getSetting('excel_collapse')) {
      $summary[] = $this->t('Excel: Collapse and hide column');
    }
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    return $this->commonSettingsSummary($summary);
  }

}
