<?php

namespace Drupal\xls_serialization_extras\Plugin\views\field;

/**
 * Processes tokens in Excel formulas and applies filters.
 */
class FormulaTokenProcessor {

  /**
   * The registered filters for processing tokens.
   *
   * @var array
   */
  private $filters = [];

  public function __construct() {
    $this->registerDefaultFilters();
  }

  /**
   * Registers a custom filter.
   */
  public function registerFilter($name, $callback) {
    $this->filters[$name] = $callback;
  }

  /**
   * Registers default filters for common operations.
   */
  private function registerDefaultFilters() {
    // Excel double-quote escape filter.
    $this->filters['escape'] = $this->filters['e'] = function ($value) {
      return str_replace('"', '""', $value);
    };

    // String filters.
    $this->filters['upper'] = function ($value) {
      return strtoupper($value);
    };
    $this->filters['lower'] = function ($value) {
      return strtolower($value);
    };
    $this->filters['capitalize'] = function ($value) {
      return ucfirst(strtolower($value));
    };
    $this->filters['title'] = function ($value) {
      return ucwords(strtolower($value));
    };

    // Number filters.
    $this->filters['currency'] = function ($value, $symbol = '$') {
      if (is_numeric($value)) {
        return $symbol . number_format($value, 2);
      }
      else {
        return 'Error: not a numeric value';
      }
    };
    $this->filters['number'] = function ($value) {
      if (is_numeric($value)) {
        return number_format($value);
      }
      else {
        return 'Error: not a numeric value';
      }
    };

    // Default filter with parameter support.
    $this->filters['default'] = function ($value, $default = '') {
      return empty($value) ? $default : $value;
    };

    // Date filter.
    $this->filters['date'] = function ($value, $format = 'Y-m-d') {
      if (is_numeric($value)) {
        return date($format, $value);
      }
      return date($format, strtotime($value));
    };

    // Length filter.
    $this->filters['length'] = function ($value) {
      return strlen($value);
    };

  }

  /**
   * Applies a filter expression to a value.
   *
   * @param mixed $value
   *   The value to filter.
   * @param string $filterExpression
   *   The filter expression, which may include multiple filters separated by
   *   pipes.
   */
  public function applyFilter($value, $filterExpression) {
    // Split by pipe to handle chained filters.
    $filters = explode('|', $filterExpression);

    // Apply each filter in sequence.
    foreach ($filters as $singleFilter) {
      $value = $this->applySingleFilter($value, trim($singleFilter));
    }

    return $value;
  }

  /**
   * Applies a single filter to a value.
   *
   * @param mixed $value
   *   The value to filter.
   * @param string $filterExpression
   *   The filter expression, which may include parameters.
   */
  private function applySingleFilter($value, $filterExpression) {
    // Parse filter and parameters.
    if (strpos($filterExpression, ':') !== FALSE) {
      [$filterName, $params] = explode(':', $filterExpression, 2);
      $filterName = trim($filterName);

      // Parse parameters (simple implementation for quoted strings and basic
      // values).
      $params = trim($params);
      if (preg_match('/^[\'"](.+)[\'"]$/', $params, $matches)) {
        // Remove quotes.
        $params = $matches[1];
      }
    }
    else {
      $filterName = trim($filterExpression);
      $params = NULL;
    }

    if (!isset($this->filters[$filterName])) {
      throw new \Exception("Unknown filter: $filterName");
    }

    $filter = $this->filters[$filterName];

    if ($params !== NULL) {
      return $filter($value, $params);
    }
    else {
      return $filter($value);
    }
  }

  /**
   * Processes a template string with token replacements and filters.
   *
   * This method replaces tokens in the format {{ tokenName|filter1|filter2 }}
   *
   * @param string $template
   *   The template string containing tokens.
   * @param array $values
   *   An associative array of values where keys are token names.
   */
  public function process($template, $values) {
    // Updated pattern to handle piped filters properly.
    $pattern = '/\{\{\s*([^}|]+)(?:\|([^}]+))?\s*}}/';

    return preg_replace_callback($pattern, function ($matches) use ($values) {
      $tokenName = trim($matches[1]);
      $filters = isset($matches[2]) ? trim($matches[2]) : NULL;

      // Get the value.
      $value = $values['{{ ' . $tokenName . ' }}'] ?? '';

      // Apply filters if present.
      if ($filters) {
        try {
          $value = $this->applyFilter($value, $filters);
        }
        catch (\Exception $e) {
          // Return original token if filter fails.
          return $matches[0];
        }
      }

      return $value;
    }, $template);
  }

  /**
   * Helper method to extract all tokens from a template.
   *
   * This method returns an array of tokens found in the template,
   * including their names and any associated filters.
   *
   * @param string $template
   *   The template string containing tokens.
   */
  public function extractTokens($template) {
    $pattern = '/\{\{\s*([^}|]+)(?:\|([^}]+))?\s*}}/';
    $matches = [];
    preg_match_all($pattern, $template, $matches, PREG_SET_ORDER);

    $result = [];
    foreach ($matches as $match) {
      $token = trim($match[1]);
      $filter = isset($match[2]) ? trim($match[2]) : NULL;

      $result[$match[0]] = [
        'token' => $token,
        'filter' => $filter,
      ];
    }

    return $result;
  }

}
