<?php

namespace Drupal\xls_serialization_extras\Plugin\Field\FieldFormatter;

/**
 * Utility class Excel formatters.
 */
class ExcelFormatterUtils {

  /**
   * Converts PHP date format patterns to Excel date format patterns.
   *
   * @param string $phpPattern
   *   PHP date format string.
   *
   * @return string
   *   Excel date format string
   */
  public static function phpToExcelDatePattern($phpPattern) {
    // Mapping of PHP date format characters to Excel format characters.
    $mapping = [
      // Day.
      // Day of the month, 2 digits with leading zeros (01 to 31)
      'd' => 'dd',
      // Day of the month without leading zeros (1 to 31)
      'j' => 'd',
      // A textual representation of a day, three letters (Mon through Sun)
      'D' => 'ddd',
      // A full textual representation of the day of the week
      // (Sunday through Saturday)
      'l' => 'dddd',
      // ISO-8601 numeric representation of the day of the week
      // (1=Monday, 7=Sunday) - Not supported in Excel.
      'N' => '',
      // English ordinal suffix for the day of the month (st, nd, rd or th)
      // Not supported in Excel.
      'S' => '',
      // Numeric representation of the day of the week (0=Sunday, 6=Saturday)
      // Not supported in Excel.
      'w' => '',
      // The day of the year (starting from 0) - not supported in Excel.
      'z' => '',

      // Week.
      // ISO-8601 week number of year - not supported in Excel.
      'W' => '',

      // Month.
      // A full textual representation of a month (January through December)
      'F' => 'mmmm',
      // Numeric representation of a month, with leading zeros (01 to 12)
      'm' => 'mm',
      // A short textual representation of a month, three letters
      // (Jan through Dec)
      'M' => 'mmm',
      // Numeric representation of a month, without leading zeros (1 to 12)
      'n' => 'm',
      // Number of days in the given month - not supported in Excel.
      't' => '',

      // Year.
      // Whether it's a leap year (1 if it is a leap year, 0 otherwise)
      // Not supported in Excel.
      'L' => '',
      // ISO-8601 week-numbering year.
      'o' => 'yyyy',
      // A full numeric representation of a year, 4 digits (1999 or 2003)
      'Y' => 'yyyy',
      // A two-digit representation of a year (99 or 03)
      'y' => 'yy',

      // Time.
      // Lowercase Ante meridiem and Post meridiem (am or pm)
      'a' => 'am/pm',
      // Uppercase Ante meridiem and Post meridiem (AM or PM)
      'A' => 'AM/PM',
      // Swatch Internet time - not supported in Excel.
      'B' => '',
      // 12-hour format of an hour without leading zeros (1 to 12)
      'g' => 'h',
      // 24-hour format of an hour without leading zeros (0 to 23)
      // Excel uses h for both
      'G' => 'h',
      // 12-hour format of an hour with leading zeros (01 to 12)
      'h' => 'hh',
      // 24-hour format of an hour with leading zeros (00 to 23)
      'H' => 'hh',
      // Minutes with leading zeros (00 to 59) - Note: conflicts with month,
      // context-dependent.
      'i' => 'mm',
      // Seconds with leading zeros (00 to 59)
      's' => 'ss',
      // Microseconds - not supported in Excel.
      'u' => '',
      // Milliseconds - not supported in Excel.
      'v' => '',

      // Timezone.
      // Timezone identifier - not supported in Excel.
      'e' => '',
      // Whether or not the date is in daylight saving time - not supported in
      // Excel.
      'I' => '',
      // Difference to Greenwich time (GMT) in hours - not supported in Excel.
      'O' => '',
      // Difference to Greenwich time (GMT) with colon - not supported in Excel.
      'P' => '',
      // Timezone abbreviation - not supported in Excel.
      'T' => '',
      // Timezone offset in seconds - not supported in Excel.
      'Z' => '',

      // Full Date/Time.
      // ISO 8601 date.
      'c' => 'yyyy-mm-dd"T"hh:mm:ss',
      // RFC 2822 formatted date.
      'r' => 'ddd, dd mmm yyyy hh:mm:ss',
      // Seconds since the Unix Epoch - not supported in Excel.
      'U' => '',
    ];

    $excelPattern = '';
    $length = strlen($phpPattern);
    $inQuotes = FALSE;
    $quoteChar = '';

    for ($i = 0; $i < $length; $i++) {
      $char = $phpPattern[$i];

      // Handle quoted strings (literal text)
      if (($char === '"' || $char === "'") && !$inQuotes) {
        $inQuotes = TRUE;
        $quoteChar = $char;
        $excelPattern .= '"';
        continue;
      }
      elseif ($char === $quoteChar && $inQuotes) {
        $inQuotes = FALSE;
        $quoteChar = '';
        $excelPattern .= '"';
        continue;
      }

      // If we're inside quotes, just copy the character.
      if ($inQuotes) {
        $excelPattern .= $char;
        continue;
      }

      // Handle escaped characters.
      if ($char === '\\' && $i + 1 < $length) {
        $nextChar = $phpPattern[$i + 1];
        $excelPattern .= '"' . $nextChar . '"';
        // Skip the next character.
        $i++;
        continue;
      }

      // Convert PHP format character to Excel format.
      if (isset($mapping[$char])) {
        $excelPattern .= $mapping[$char];
      }
      else {
        // For common separators and literal characters, don't quote them.
        if (in_array($char, ['-', '/', ':', ' ', ',', '.', '(', ')', '[', ']'])) {
          $excelPattern .= $char;
        }
        else {
          // Only quote special characters that might conflict.
          $excelPattern .= '"' . $char . '"';
        }
      }
    }

    return $excelPattern;
  }

  /**
   * Helper function to handle time format conversion with context awareness.
   *
   * This addresses the conflict between 'i' (minutes) and 'm' (month) in Excel.
   *
   * @param string $phpPattern
   *   PHP date format string.
   *
   * @return string
   *   Excel date format string with proper time context
   */
  public static function phpToExcelDatePatternAdvanced($phpPattern) {
    $result = static::phpToExcelDatePattern($phpPattern);

    // Post-process to handle minute/month conflicts.
    // If we have time components (h or s), convert 'mm' to minutes where
    // appropriate.
    $hasHour = (strpos($result, 'h') !== FALSE);
    $hasSecond = (strpos($result, 'ss') !== FALSE);

    if ($hasHour || $hasSecond) {
      // Replace 'mm' with ':mm' for minutes when in time context
      // This is a simplified approach - in practice, you might need
      // more sophisticated logic.
      $result = preg_replace('/(?<=h)mm/', ':mm', $result);
      $result = preg_replace('/mm(?=:ss)/', ':mm', $result);
    }

    return $result;
  }

}
