<?php

namespace Drupal\Tests\xray_audit_insight\Kernel\Form;

use Drupal\xray_audit_insight\Plugin\XrayAuditInsightPluginManager;
use Drupal\xray_audit_insight\Plugin\XrayAuditInsightPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormState;
use Drupal\KernelTests\ConfigFormTestBase;
use Drupal\xray_audit_insight\Form\SettingsForm;

/**
 * Tests the xray_audit_insight SettingsForm.
 *
 * @group xray_audit_insight
 * @coversDefaultClass \Drupal\xray_audit_insight\Form\SettingsForm
 */
class SettingsFormTest extends ConfigFormTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'xray_audit',
    'xray_audit_insight',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\xray_audit_insight\Form\SettingsForm
   */
  protected $form;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The plugin manager service.
   *
   * @var \Drupal\xray_audit_insight\Plugin\XrayAuditInsightPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['xray_audit_insight']);

    // Get services.
    $this->configFactory = $this->container->get('config.factory');
    $this->pluginManager = $this->container->get('plugin_manager.xray_audit_insight');

    // Create form instance.
    $this->form = SettingsForm::create($this->container);
  }

  /**
   * Tests form instantiation through dependency injection.
   *
   * @covers ::create
   */
  public function testFormInstantiation() {
    // Arrange & Act: Get form from container.
    $form = SettingsForm::create($this->container);

    // Assert: Form is instantiated correctly.
    $this->assertInstanceOf(
      SettingsForm::class,
      $form,
      'Form should be properly instantiated'
    );
  }

  /**
   * Tests that form implements FormInterface.
   */
  public function testFormImplementsFormInterface() {
    // Assert: Form implements FormInterface.
    $this->assertInstanceOf(
      FormInterface::class,
      $this->form,
      'Form should implement FormInterface'
    );
  }

  /**
   * Tests getFormId returns the correct form ID.
   *
   * @covers ::getFormId
   */
  public function testGetFormId() {
    // Act: Get form ID.
    $form_id = $this->form->getFormId();

    // Assert: Form ID is correct.
    $this->assertEquals(
      'xray_audit_insight.settings',
      $form_id,
      'Form ID should be xray_audit_insight.settings'
    );
  }

  /**
   * Tests that getFormId returns a string.
   *
   * @covers ::getFormId
   */
  public function testGetFormIdReturnsString() {
    // Act: Get form ID.
    $form_id = $this->form->getFormId();

    // Assert: Form ID is a string.
    $this->assertIsString($form_id, 'Form ID should be a string');
    $this->assertNotEmpty($form_id, 'Form ID should not be empty');
  }

  /**
   * Tests getEditableConfigNames returns correct config names.
   *
   * @covers ::getEditableConfigNames
   */
  public function testGetEditableConfigNames() {
    // Act: Get editable config names using reflection.
    $reflection = new \ReflectionClass($this->form);
    $method = $reflection->getMethod('getEditableConfigNames');
    $method->setAccessible(TRUE);
    $config_names = $method->invoke($this->form);

    // Assert: Config names are correct.
    $this->assertIsArray($config_names, 'Config names should be an array');
    $this->assertContains('xray_audit_insight.settings', $config_names, 'Should include xray_audit_insight.settings');
  }

  /**
   * Tests that buildForm returns proper form array structure.
   *
   * @covers ::buildForm
   */
  public function testBuildFormStructure() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Form has required elements.
    $this->assertArrayHasKey('insight_switch', $form_array, 'Form should have insight_switch container');
    $this->assertArrayHasKey('actions', $form_array, 'Form should have actions');
  }

  /**
   * Tests that insight_switch container has correct structure.
   *
   * @covers ::buildForm
   */
  public function testInsightSwitchContainerStructure() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Insight switch container has correct properties.
    $this->assertEquals('container', $form_array['insight_switch']['#type'], 'Insight switch should be a container');
    $this->assertArrayHasKey('title', $form_array['insight_switch'], 'Insight switch should have title element');
    $this->assertArrayHasKey('description', $form_array['insight_switch'], 'Insight switch should have description element');
  }

  /**
   * Tests that title element has correct structure.
   *
   * @covers ::buildForm
   */
  public function testTitleElementStructure() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Title has correct structure.
    $this->assertEquals('html_tag', $form_array['insight_switch']['title']['#type'], 'Title should be html_tag type');
    $this->assertEquals('h4', $form_array['insight_switch']['title']['#tag'], 'Title should be h4 tag');
    $this->assertNotEmpty($form_array['insight_switch']['title']['#value'], 'Title should have value');
  }

  /**
   * Tests that description element has correct structure.
   *
   * @covers ::buildForm
   */
  public function testDescriptionElementStructure() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Description has correct structure.
    $this->assertEquals('html_tag', $form_array['insight_switch']['description']['#type'], 'Description should be html_tag type');
    $this->assertEquals('p', $form_array['insight_switch']['description']['#tag'], 'Description should be p tag');
    $this->assertNotEmpty($form_array['insight_switch']['description']['#value'], 'Description should have value');
  }

  /**
   * Tests that form integrates with insight plugins.
   *
   * @covers ::buildForm
   * @covers ::getInsights
   */
  public function testFormIntegratesWithInsightPlugins() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Get available plugin definitions.
    $definitions = $this->pluginManager->getDefinitions();

    // Assert: Form should build wrapper for each plugin.
    if (!empty($definitions)) {
      foreach ($definitions as $plugin_id => $definition) {
        $wrapper_key = $plugin_id . '_wrapper';
        // The form may or may not have wrappers depending on plugins.
        // Just verify the insight_switch container exists.
        $this->assertArrayHasKey('insight_switch', $form_array, 'Form should have insight_switch for plugins');
      }
    }
  }

  /**
   * Tests form submission saves configuration.
   *
   * @covers ::submitForm
   */
  public function testFormSubmissionSavesConfiguration() {
    // Arrange: Create form state with values.
    $form_state = new FormState();
    $form_state->setValue('excluded_views', ['test_view']);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Configuration is saved.
    $config = $this->configFactory->get('xray_audit_insight.settings');
    $excluded_views = $config->get('excluded_views');
    $this->assertEquals(['test_view'], $excluded_views, 'Excluded views should be saved');
  }

  /**
   * Tests form submission handles excluded views.
   *
   * @covers ::submitForm
   */
  public function testFormSubmissionHandlesExcludedViews() {
    // Arrange: Create form state with multiple excluded views.
    $form_state = new FormState();
    $excluded_views = ['view1', 'view2', 'view3'];
    $form_state->setValue('excluded_views', $excluded_views);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: All excluded views are saved.
    $config = $this->configFactory->get('xray_audit_insight.settings');
    $saved_views = $config->get('excluded_views');
    $this->assertEquals($excluded_views, $saved_views, 'All excluded views should be saved');
  }

  /**
   * Tests form submission skips excluded views when not set.
   *
   * @covers ::submitForm
   */
  public function testFormSubmissionSkipsExcludedViewsWhenNotSet() {
    // Arrange: Create form state without excluded_views.
    $form_state = new FormState();

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Config is not set for excluded_views when value is null.
    // This test verifies the if condition works correctly.
    $this->assertTrue(TRUE, 'Form should handle missing excluded_views gracefully');
  }

  /**
   * Tests form titles are translatable.
   *
   * @covers ::buildForm
   */
  public function testFormTitlesAreTranslatable() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Title value is TranslatableMarkup.
    $this->assertInstanceOf(
      TranslatableMarkup::class,
      $form_array['insight_switch']['title']['#value'],
      'Title should be translatable'
    );
  }

  /**
   * Tests form descriptions are translatable.
   *
   * @covers ::buildForm
   */
  public function testFormDescriptionsAreTranslatable() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Description value is TranslatableMarkup.
    $this->assertInstanceOf(
      TranslatableMarkup::class,
      $form_array['insight_switch']['description']['#value'],
      'Description should be translatable'
    );
  }

  /**
   * Tests that form has submit button from parent.
   *
   * @covers ::buildForm
   */
  public function testFormHasSubmitButton() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Form has actions with submit button (from parent class).
    $this->assertArrayHasKey('actions', $form_array, 'Form should have actions');
    $this->assertArrayHasKey('submit', $form_array['actions'], 'Form should have submit button');
  }

  /**
   * Tests MODULE_SETTINGS constant is defined correctly.
   *
   * @covers ::buildForm
   */
  public function testModuleSettingsConstant() {
    // Act & Assert: Check MODULE_SETTINGS constant.
    $this->assertEquals(
      'xray_audit_insight.settings',
      SettingsForm::MODULE_SETTINGS,
      'MODULE_SETTINGS constant should be xray_audit_insight.settings'
    );
  }

  /**
   * Tests form updates existing configuration values.
   *
   * @covers ::submitForm
   */
  public function testFormUpdatesExistingConfiguration() {
    // Arrange: Set initial config values.
    $config = $this->configFactory->getEditable('xray_audit_insight.settings');
    $config->set('excluded_views', ['old_view1', 'old_view2']);
    $config->save();

    // Create form state with new values.
    $form_state = new FormState();
    $form_state->setValue('excluded_views', ['new_view1', 'new_view2', 'new_view3']);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Configuration is updated.
    $config = $this->configFactory->get('xray_audit_insight.settings');
    $excluded_views = $config->get('excluded_views');
    $this->assertEquals(['new_view1', 'new_view2', 'new_view3'], $excluded_views, 'Excluded views should be updated');
  }

  /**
   * Tests getInsights returns array of insight plugins.
   *
   * @covers ::getInsights
   */
  public function testGetInsightsReturnsArray() {
    // Arrange: Use reflection to access protected method.
    $reflection = new \ReflectionClass($this->form);
    $method = $reflection->getMethod('getInsights');
    $method->setAccessible(TRUE);

    // Act: Get insights.
    $insights = $method->invoke($this->form);

    // Assert: Insights is an array.
    $this->assertIsArray($insights, 'getInsights should return an array');
  }

  /**
   * Tests getInsights creates plugin instances.
   *
   * @covers ::getInsights
   */
  public function testGetInsightsCreatesPluginInstances() {
    // Arrange: Use reflection to access protected method.
    $reflection = new \ReflectionClass($this->form);
    $method = $reflection->getMethod('getInsights');
    $method->setAccessible(TRUE);

    // Act: Get insights.
    $insights = $method->invoke($this->form);

    // Assert: Each insight is a plugin instance.
    foreach ($insights as $plugin_id => $insight) {
      $this->assertInstanceOf(
        XrayAuditInsightPluginInterface::class,
        $insight,
        "Insight {$plugin_id} should be a plugin instance"
      );
    }
  }

  /**
   * Tests form submission calls plugin submitInsightSettings.
   *
   * @covers ::submitForm
   */
  public function testFormSubmissionCallsPluginSubmitInsightSettings() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Form completes without errors.
    // The actual plugin submission is tested in plugin tests.
    $this->assertEmpty($form_state->getErrors(), 'Form submission should complete without errors');
  }

  /**
   * Tests form builds without errors.
   *
   * @covers ::buildForm
   */
  public function testFormBuildsWithoutErrors() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Form is an array.
    $this->assertIsArray($form_array, 'Form should build as an array');
    $this->assertNotEmpty($form_array, 'Form should not be empty');
  }

  /**
   * Tests form renders correctly through form builder.
   *
   * @covers ::buildForm
   */
  public function testFormRendersCorrectlyThroughFormBuilder() {
    // Arrange: Get form builder.
    $form_builder = $this->container->get('form_builder');

    // Act: Get form through form builder.
    $form = $form_builder->getForm($this->form);

    // Assert: Form is built correctly.
    $this->assertIsArray($form, 'Form should be an array');
    $this->assertArrayHasKey('insight_switch', $form, 'Form should have insight_switch');
    $this->assertArrayHasKey('actions', $form, 'Form should have actions');
  }

  /**
   * {@inheritdoc}
   *
   * Override parent test to provide form-specific values.
   */
  public function testConfigForm(): void {
    // Arrange: Build form.
    $form_builder = $this->container->get('form_builder');

    // Act: Get form.
    $form = $form_builder->getForm($this->form);

    // Assert: Form is built.
    $this->assertNotEmpty($form);

    // Arrange: Set values for submission.
    $values = [
      'op' => 'Save configuration',
    ];

    // Act: Submit form.
    $form_state = (new FormState())->setValues($values);
    $form_builder->submitForm($this->form, $form_state);

    // Assert: No validation errors.
    $errors = $form_state->getErrors();
    $this->assertEmpty($errors, 'No validation errors should occur');
  }

  /**
   * Tests form handles empty plugin list.
   *
   * @covers ::buildForm
   * @covers ::getInsights
   */
  public function testFormHandlesEmptyPluginList() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form (even if no plugins are available).
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Form is still built successfully.
    $this->assertIsArray($form_array, 'Form should build even with empty plugin list');
    $this->assertArrayHasKey('insight_switch', $form_array, 'Form should have insight_switch container');
  }

  /**
   * Tests form handles plugin manager correctly.
   *
   * @covers ::create
   */
  public function testFormHandlesPluginManagerCorrectly() {
    // Arrange: Create form through DI.
    $form = SettingsForm::create($this->container);

    // Act: Check if form has plugin manager.
    $reflection = new \ReflectionClass($form);
    $property = $reflection->getProperty('pluginManager');
    $property->setAccessible(TRUE);
    $plugin_manager = $property->getValue($form);

    // Assert: Plugin manager is set.
    $this->assertNotNull($plugin_manager, 'Plugin manager should be set');
    $this->assertInstanceOf(
      XrayAuditInsightPluginManager::class,
      $plugin_manager,
      'Plugin manager should be correct instance'
    );
  }

}
