<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric;

use Drupal\Core\Entity\Query\QueryAggregateInterface;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "queries_data_blocks",
 *   label = @Translation("Blocks reports"),
 *   description = @Translation("Metrics about block entities."),
 *   group = "content_metric",
 *   sort = 2,
 *   local_task = 1,
 *   operations = {
 *     "block_count_type" = {
 *          "label" = "Blocks grouped by type",
 *          "description" = "Number of Block grouped by type.",
 *           "download" = TRUE
 *       },
 *     "block_count_type_langcode" = {
 *          "label" = "Blocks grouped by type and language",
 *          "description" = "Number of Block grouped by type and language.",
 *           "download" = TRUE
 *      },
 *    },
 *   dependencies = {"block_content"},
 * )
 *
 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
 */
final class XrayAuditQueryTaskBlockPlugin extends XrayAuditTaskPluginBase {

  /**
   * Get base query.
   *
   * @return \Drupal\Core\Entity\Query\QueryAggregateInterface
   *   Query.
   */
  protected function getBaseQuery(): QueryAggregateInterface {
    $alias = 'count';
    $query = $this->entityTypeManager->getStorage('block_content')->getAggregateQuery()
      ->accessCheck(FALSE)
      ->currentRevision()
      ->groupBy('type')
      ->aggregate('id', 'COUNT', NULL, $alias)
      ->sort('type');

    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'block_count_type':
        return [
          $this->t('Type'),
          $this->t('Label'),
          $this->t('Total'),
        ];

      case 'block_count_type_langcode':
        return [
          $this->t('Type'),
          $this->t('Label'),
          $this->t('Langcode'),
          $this->t('Total'),
        ];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'block_count_type':
        return $this->getBlockCountTypeRows();

      case 'block_count_type_langcode':
        return $this->getBlockCountTypeLangcodeRows();
    }
    return [];
  }

  /**
   * Gets the rows for the 'block_count_type' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getBlockCountTypeRows(): array {
    $table_content = [];
    $block_content_types = $this->entityTypeManager->getStorage('block_content_type')->loadMultiple();
    $result = $this->getBaseQuery()->execute();

    /** @var mixed[] $row */
    foreach ($result as $row) {
      if (empty($row['type'])) {
        continue;
      }
      $table_content[] = [$row['type'], $block_content_types[$row['type']]->label(), $row['count']];
    }
    return $table_content;
  }

  /**
   * Gets the rows for the 'block_count_type_langcode' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getBlockCountTypeLangcodeRows(): array {
    $table_content = [];
    $block_content_types = $this->entityTypeManager->getStorage('block_content_type')->loadMultiple();
    $result = $this->getBaseQuery()
      ->groupBy('langcode')
      ->sort('langcode')
      ->execute();

    /** @var mixed[] $row */
    foreach ($result as $row) {
      if (empty($row['type'])) {
        continue;
      }
      $table_content[] = [$row['type'], $block_content_types[$row['type']]->label(), $row['langcode'], $row['count']];
    }
    return $table_content;
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $data = [
      'header_table' => $this->getHeaders($operation),
      'results_table' => $this->getRows($operation),
    ];

    $this->pluginRepository->setCacheTagsInv($cid, $data, ['block_content_list']);
    return $data;
  }

}
