<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric;

use Drupal\Component\Plugin\DependentPluginInterface;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FormatterPluginManager;
use Drupal\Core\Link;
use Drupal\field_ui\FieldUI;
use Drupal\image\ImageStyleStorageInterface;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Drupal\xray_audit\Services\EntityArchitectureInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an image styles report.
 *
 * @XrayAuditTaskPlugin(
 *   id = "queries_data_image_styles",
 *   label = @Translation("Image Styles Report"),
 *   description = @Translation("Reports on the image styles used in the site"),
 *   group = "content_metric",
 *   sort = 7,
 *   local_task = 1,
 *   operations = {
 *      "image_styles_usage" = {
 *          "label" = "Image Styles Usage",
 *          "description" = "Lists image styles and where they are used.",
 *          "download" = FALSE
 *       },
 *      "responsive_image_styles_usage" = {
 *          "label" = "Responsive Image Styles Usage",
 *          "description" = "Lists responsive image styles and where they are used.",
 *          "download" = FALSE
 *      },
 *      "ckeditor_image_styles_usage" = {
 *          "label" = "CKEditor Image Styles",
 *          "description" = "List of image styles that can be used from CKEditor.",
 *          "download" = FALSE
 *       },
 *    },
 *   dependencies = {"media", "image", "responsive_image"},
 * )
 */
class XrayAuditQueryTaskImageStylesPlugin extends XrayAuditTaskPluginBase {

  /**
   * The entity architecture service.
   *
   * @var \Drupal\xray_audit\Services\EntityArchitectureInterface
   */
  protected EntityArchitectureInterface $entityArchitecture;

  /**
   * The bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $bundleInfo;

  /**
   * The image style storage.
   *
   * @var \Drupal\image\ImageStyleStorageInterface
   */
  protected ImageStyleStorageInterface $imageStyleStorage;

  /**
   * The responsive image style storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $responsiveImageStyleStorage;

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The field formatter manager service.
   *
   * @var \Drupal\Core\Field\FormatterPluginManager
   */
  protected FormatterPluginManager $fieldFormatterManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityArchitecture = $container->get('xray_audit.entity_architecture');
    $instance->bundleInfo = $container->get('entity_type.bundle.info');
    $instance->imageStyleStorage = $instance->entityTypeManager->getStorage('image_style');
    $instance->responsiveImageStyleStorage = $instance->entityTypeManager->getStorage('responsive_image_style');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->fieldFormatterManager = $container->get('plugin.manager.field.formatter');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;
    $data = $this->pluginRepository->getCachedData($cid);
    $cache_tags = [];

    if (!empty($data) && is_array($data)) {
      return $data;
    }

    switch ($operation) {
      case 'image_styles_usage':
        $data = $this->getImageStylesUsage();
        $cache_tags = [
          'config:entity_view_display_list',
          'config:responsive_image_style_list',
        ];
        break;

      case 'responsive_image_styles_usage':
        $data = $this->getResponsiveImageStylesUsage();
        $cache_tags = ['config:entity_view_display_list'];
        break;

      case 'ckeditor_image_styles_usage':
        $data = $this->getCKEditorImageStyles();
        break;
    }

    $this->pluginRepository->setCacheTagsInv($cid, $data, $cache_tags);

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function buildDataRenderArray(array $data, string $operation = '') {
    // @todo Improve method, construct renderable by calling different methods.
    $renderable = NULL;
    $allowed_operations = [
      'image_styles_usage',
      'responsive_image_styles_usage',
    ];

    if (in_array($operation, $allowed_operations)) {

      $renderable = [
        '#type' => 'container',
        'visibility_toggler' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Hide elements with no usages.'),
          '#attributes' => ['xa-visibility-toggler' => ''],
        ],
        'wrappers' => [],
        '#attached' => [
          'library' => [
            'xray_audit/visibility_toggler',
          ],
        ],
      ];

      foreach ($data as $id => $item) {
        $table_data = $item['data'];
        $item['link']->getUrl()->setOption('attributes', ['target' => '_blank']);
        $renderable_link = $item['link']->toRenderable();
        $renderable_link['#attributes']['style'] = [
          'text-transform: lowercase;',
          'font-size: 0.9em;',
        ];
        $renderable_link['#title'] = "({$renderable_link['#title']})";

        $renderable['wrappers'][$id] = [
          '#type' => 'container',
          'elements' => [
            '#type' => 'details',
            '#title' => [
              'label' => [
                '#markup' => (string) $item['label'] . ' ',
              ],
              'link' => $renderable_link,
            ],
            '#attributes' => ['xa-results-wrapper' => ''],
            '#open' => FALSE,
            'display_modes_usage' => [
              '#type' => 'details',
              '#title' => $this->t('Display Modes Usage'),
              '#open' => TRUE,
              'table' => $this->buildDisplayModeTable($table_data),
            ],
          ],
        ];

        if ($operation == 'image_styles_usage') {
          $elements = &$renderable['wrappers'][$id]['elements'];
          $elements['responsive_image_styles_usage'] = [
            '#type' => 'details',
            '#title' => $this->t('Responsive image styles usages'),
            '#open' => FALSE,
            'table' => [
              '#type' => 'table',
              '#header' => $this->getResponsiveImageStylesUsagesDataHeaders(),
              '#rows' => $table_data['responsive_img_styles_data']['rows'] ?? [],
              '#empty' => $this->t('No usages found.'),
            ],
          ];
        }
      }

    }

    return $renderable ?? parent::buildDataRenderArray($data, $operation);
  }

  /**
   * Retrieves the image styles usage report.
   *
   * @return array
   *   The image styles usage data including headers and results.
   */
  protected function getImageStylesUsage() {
    $image_styles_data = [];
    $image_styles_ids = $this->getImageStylesIds();
    $view_modes = $this->getImageStylesEntityViewDisplayUsage($image_styles_ids);
    // Image style usage on responsive image styles.
    $isu_on_ris = $this->getImageStyleUsagesInResponsiveImageStyles();

    foreach ($image_styles_ids as $id) {
      $image_style = $this->imageStyleStorage->load($id);

      if ($image_style instanceof EntityInterface) {
        $image_styles_data[$id] = [
          'id' => $id,
          'label' => $image_style->label(),
          'link' => Link::fromTextAndUrl($id, $image_style->toUrl()),
          'data' => [
            'view_modes_data' => [
              'rows' => $view_modes[$id] ?? [],
            ],
            'responsive_img_styles_data' => [
              'rows' => $isu_on_ris[$id] ?? [],
            ],
          ],
        ];
      }

    }

    return $image_styles_data;
  }

  /**
   * Retrieves the usage data of responsive image styles.
   *
   * This method gathers information about responsive image styles, including
   * their IDs, labels, links, and associated view modes data.
   * It loads the responsive image styles, retrieves their labels and URLs,
   * and compiles the data into an array.
   *
   * @return array
   *   An associative array containing the usage of responsive image styles.
   */
  protected function getResponsiveImageStylesUsage() {
    $responsive_image_styles_data = [];
    $responsive_image_styles_ids = $this->getResponsiveImageStyleIds();
    $view_modes = $this->getResponsiveImageStylesEntityViewDisplayUsage($responsive_image_styles_ids);

    foreach ($responsive_image_styles_ids as $id) {
      $image_style = $this->responsiveImageStyleStorage->load($id);

      if ($image_style instanceof EntityInterface) {
        $responsive_image_styles_data[$id] = [
          'id' => $id,
          'label' => $image_style->label(),
          'link' => Link::fromTextAndUrl($id, $image_style->toUrl()),
          'data' => [
            'view_modes_data' => [
              'rows' => $view_modes[$id] ?? [],
            ],
          ],
        ];
      }
    }

    return $responsive_image_styles_data;
  }

  /**
   * Retrieves the usages of image styles within responsive image styles.
   *
   * @return array
   *   An array of usages, keyed by image style ID, containing information about
   *   where each image style is used in responsive image styles.
   */
  protected function getImageStyleUsagesInResponsiveImageStyles() {
    $usages = [];
    /** @var \Drupal\responsive_image\ResponsiveImageStyleInterface[] $responsive_image_styles */
    $responsive_image_styles = $this->responsiveImageStyleStorage->loadMultiple();

    foreach ($responsive_image_styles as $responsive_image_style) {

      $mappings = $responsive_image_style->getImageStyleMappings();
      $url = $responsive_image_style->toUrl();
      $link = Link::fromTextAndUrl((string) $responsive_image_style->id(), $url);

      foreach ($mappings as $mapping) {
        $image_styles = $mapping['image_mapping']['sizes_image_styles'] ?? (array) $mapping['image_mapping'];

        foreach ($image_styles as $image_style) {
          $usages[$image_style][] = [
            'id' => $link,
            'label' => $responsive_image_style->label(),
            'image_mapping_type' => $mapping['image_mapping_type'],
            'breakpoint' => $mapping['breakpoint_id'] ?? '',
            'multiplier' => $mapping['multiplier'] ?? '',
          ];
        }
      }
    }

    return $usages;
  }

  /**
   * Builds a table for displaying view modes data.
   *
   * @param array $data
   *   An associative array containing the view modes data to be displayed.
   *
   *   - view_modes_data: An array containing:
   *     - header: (optional) An array of table header labels.
   *     - rows: (optional) An array of table rows.
   *
   * @return array
   *   A render array representing the table with view modes data.
   */
  protected function buildDisplayModeTable(array $data) {
    return [
      '#type' => 'table',
      '#header' => $this->getDisplayModeDataHeaders(),
      '#rows' => $data['view_modes_data']['rows'] ?? [],
      '#empty' => $this->t('No usages found.'),
    ];
  }

  /**
   * Retrieves all image styles.
   *
   * @return array
   *   An array of image style entities ids.
   */
  protected function getImageStylesIds(): array {
    $query = $this->imageStyleStorage->getQuery();
    $query->accessCheck(FALSE);
    $ids = $query->execute();

    return $ids;
  }

  /**
   * Retrieves all image styles.
   *
   * @return array
   *   An array of responsive image style entities ids.
   */
  protected function getResponsiveImageStyleIds(): array {
    $query = $this->responsiveImageStyleStorage->getQuery();
    $query->accessCheck(FALSE);
    $ids = $query->execute();

    return $ids;
  }

  /**
   * Retrieves the headers for the view modes data table.
   *
   * @return array
   *   An array representing the headers for the display modes data table.
   *
   *   The headers include:
   *   - Entity: The entity type.
   *   - Type: The type of the entity.
   *   - Field Name: The name of the field.
   *   - View Mode: The view mode of the entity.
   *   - Formatter: The formatter used for the field.
   *   - Setting Path: The path to the settings.
   */
  protected function getDisplayModeDataHeaders() {
    return [
      $this->t('Entity'),
      $this->t('Bundle'),
      $this->t('Field type'),
      $this->t('Field Name'),
      $this->t('View Mode'),
      $this->t('Formatter'),
    ];
  }

  /**
   * Retrieves the headers for the responsive image styles usages data table.
   */
  protected function getResponsiveImageStylesUsagesDataHeaders() {
    return [
      $this->t('ID'),
      $this->t('Label'),
      $this->t('Image Mapping type'),
      $this->t('Breakpoints'),
      $this->t('Multiplier'),
    ];
  }

  /**
   * Retrieves entity view displays based on entity type and bundle.
   *
   * @param string $entity_type
   *   (optional) The entity type to filter by.
   * @param string $bundle
   *   (optional) The bundle to filter by.
   *
   * @return array
   *   An array of entity view display IDs.
   */
  protected function getEntityViewDisplays(string $entity_type = '', string $bundle = '') {
    $query = $this->entityTypeManager->getStorage('entity_view_display')->getQuery();

    if (!empty($entity_type)) {
      $query->condition('targetEntityType', $entity_type);
    }

    if (!empty($bundle)) {
      $query->condition('bundle', $bundle);
    }

    $result = $query->execute();

    return $result;
  }

  /**
   * Retrieves the usage of entity view displays for the given image style IDs.
   *
   * This method queries the database to find out where the specified image
   * styles are being used within entity view displays. It returns an array
   * containing the usage information for each image style ID provided.
   *
   * @param array $image_styles_ids
   *   An array of image style IDs to check for usage in entity view displays.
   *
   * @return array
   *   An associative array where the keys are image style IDs and the values
   *   are arrays of entity view display usage information.
   */
  protected function getImageStylesEntityViewDisplayUsage(array $image_styles_ids) {
    $data = [];
    $view_displays_ids = $this->getEntityViewDisplays();
    $view_displays = $this->entityTypeManager->getStorage('entity_view_display')->loadMultiple($view_displays_ids);
    $image_styles = $this->imageStyleStorage->loadMultiple($image_styles_ids);
    $pattern = '^(?!responsive_([^_]+_)?style($|_.+)).+_style($|_.+)';
    $data = $this->buildEntityViewDisplayUsageData($image_styles, $pattern, ...$view_displays);

    return $data;
  }

  /**
   * Retrieves the usage of responsive image styles in entity view displays.
   *
   * @param array $responsive_image_styles_ids
   *   An array of responsive image style IDs.
   *
   * @return array
   *   An array containing the usage data of responsive image styles in entity
   *   view displays.
   */
  protected function getResponsiveImageStylesEntityViewDisplayUsage(array $responsive_image_styles_ids) {
    $data = [];
    $view_displays_ids = $this->getEntityViewDisplays();
    $view_displays = $this->entityTypeManager->getStorage('entity_view_display')->loadMultiple($view_displays_ids);
    $responsive_image_styles = $this->responsiveImageStyleStorage->loadMultiple($responsive_image_styles_ids);
    $pattern = '^responsive_([^_]+_)?style($|_.+)';
    $data = $this->buildEntityViewDisplayUsageData($responsive_image_styles, $pattern, ...$view_displays);

    return $data;
  }

  /**
   * Retrieves all entity fields that can contain image styles.
   *
   * This method combines image fields and entity reference fields, as both
   * can potentially utilize image styles.
   *
   * @return array
   *   An array of allowed entity fields.
   */
  protected function getAllowedEntityFields(): array {
    $image_fields = $this->entityFieldManager->getFieldMapByFieldType('image');
    $entity_reference_fields = $this->entityFieldManager->getFieldMapByFieldType('entity_reference');

    return array_map(fn($item) => array_keys($item), array_merge_recursive($image_fields, $entity_reference_fields));
  }

  /**
   * Retrieves the image styles that are allowed in CKEditor.
   *
   * This method identifies the view modes that are enabled for media embedding
   * in CKEditor configurations.
   *
   * @return array
   *   Array containing the table headers and the results of allowed view modes.
   */
  public function getCkEditorImageStyles() {
    $headerTable = [$this->t('Formatter type'), $this->t('View Mode')];
    $resultTable = [];

    $formats = $this->entityTypeManager->getStorage('filter_format')->loadMultiple();
    $used_view_modes = [];

    // Extract allowed view modes for CKEditor media embedding.
    foreach ($formats as $format) {
      $filters = $format->get('filters');
      if (isset($filters['media_embed']) && !empty($filters['media_embed']['settings']['allowed_view_modes'])) {
        foreach ($filters['media_embed']['settings']['allowed_view_modes'] as $mode) {
          $used_view_modes[$mode][] = $format->label();
        }
      }
    }

    // Format results for display.
    foreach ($used_view_modes as $mode => $formats) {
      $resultTable[] = [implode(', ', $formats), $mode];
    }

    return [
      'header_table' => $headerTable,
      'results_table' => $resultTable,
    ];
  }

  /**
   * Builds usage data for image styles in entity view displays.
   *
   * This method iterates through the provided entity view displays and
   * extracts the usage data of image styles based on the provided image
   * styles and regex pattern.
   *
   * @param array $image_styles
   *   An array of image style entities to check for usage.
   * @param string $regex_pattern
   *   The regex pattern to match image style settings.
   * @param \Drupal\Core\Entity\Display\EntityViewDisplayInterface ...$view_displays
   *   A variable number of entity view display interfaces to check.
   *
   * @return array
   *   An array of usage data for the specified image styles.
   */
  protected function buildEntityViewDisplayUsageData(array $image_styles, string $regex_pattern, EntityViewDisplayInterface ...$view_displays) {
    $data = [];

    if (empty($image_styles)) {
      return $data;
    }

    foreach ($view_displays as $view_display) {
      $entity_type_id = $view_display->getTargetEntityTypeId();
      $bundle = $view_display->getTargetBundle();
      $components = $view_display->getComponents();

      foreach ($components as $field_name => $component) {
        $component_data_usage = $this->buildEntityViewDisplayComponentUsage($component, $image_styles, $field_name, $entity_type_id, $bundle, $view_display->getMode(), $regex_pattern);
        $data = array_merge_recursive($data, $component_data_usage);

      }
    }

    return $data;
  }

  /**
   * Builds the usage data for the given component.
   *
   * This method analyzes a specific component of an entity view display
   * to determine if it utilizes any of the provided image styles.
   *
   * @param array $component
   *   The component data.
   * @param array $image_styles
   *   An array of image style entities to check for usage.
   * @param string $field_name
   *   The name of the field.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle
   *   The bundle name.
   * @param string $view_mode
   *   The view mode.
   * @param string $pattern
   *   The regex pattern to match image style settings.
   *
   * @return array
   *   An array of usage data.
   */
  protected function buildEntityViewDisplayComponentUsage(
    array $component,
    array $image_styles,
    string $field_name,
    string $entity_type_id,
    string $bundle,
    string $view_mode,
    string $pattern,
  ) {

    $data = [];
    $settings_data_usage = [];
    $image_styles_config_dependency_names = array_map(fn($image_style) => $image_style->getConfigDependencyName(), $image_styles);
    $image_styles_ids = $image_styles_ids = array_keys($image_styles_config_dependency_names);
    $field_definition = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle)[$field_name] ?? NULL;
    $field_type = '';

    // First we try to get image styles dependencies from formatter.
    if ($field_definition instanceof FieldDefinitionInterface) {
      $field_formatter_dependencies = $this->getViewModeFieldFormatterDependencies($component, $view_mode, $field_definition);
      $field_type = $field_definition->getType();
    }

    if (!empty($field_formatter_dependencies['config'])) {
      $intersect_config = array_intersect($image_styles_config_dependency_names, $field_formatter_dependencies['config']);
      // @todo find the setting key.
      $settings_data_usage = $intersect_config;
    }

    // If dependencies are not properly constructed in formatter let's make
    // a deep iteration over all the settings keys, having into account
    // multidimensional.
    if (empty($settings_data_usage) && isset($component['settings']) && is_array($component['settings'])) {

      // Regex patterns are used in keys to avoid any conflict with other
      // keys that might have an ID identical to the image style or
      // responsive image style. For example, having a view mode called
      // "medium" while also having an image style called "medium" would
      // count as a match if we were just checking the style image ID.
      $settings_data_usage = (array) $this->getUsagesInComponentSettings($component['settings'], $image_styles_ids, $pattern);
    }

    if (!empty($settings_data_usage)) {
      $link = $this->generateDisplayModeLink($view_mode, $entity_type_id, $bundle);

      foreach ($settings_data_usage as $image_style_id => $setting_key_path) {
        $data[$image_style_id][] = [
          'entity_type' => $entity_type_id,
          'bundle' => $bundle,
          'field_type' => $field_type,
          'field_name' => $field_name,
          'view_mode' => $link,
          'formatter' => $component['type'] ?? '',
        ];
      }
    }

    return $data;
  }

  /**
   * Retrieves the dependencies for a field formatter in a specific view mode.
   *
   * This method calculates the dependencies of a field formatter used in a
   * specific view mode. These dependencies can include configuration entities,
   * content entities, or other modules.
   *
   * @param array $component
   *   The component configuration array.
   * @param string $view_mode
   *   The view mode.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   *
   * @return array
   *   An array of dependencies for the field formatter.
   */
  protected function getViewModeFieldFormatterDependencies(array $component, string $view_mode, FieldDefinitionInterface $field_definition) {

    $dependencies = [];

    if (!empty($component['type'])) {
      $formatter = $this->fieldFormatterManager->getInstance([
        'field_definition' => $field_definition,
        'view_mode' => $view_mode,
        'prepare' => FALSE,
        'configuration' => $component,
      ]);

      if ($formatter instanceof DependentPluginInterface) {
        $dependencies = $formatter->calculateDependencies();
      }
    }

    return $dependencies;
  }

  /**
   * Retrieves the image styles usages in component settings.
   *
   * This method recursively traverses the provided settings array and
   * identifies the usage of image styles based on the provided image
   * style IDs and regex pattern.
   * It returns an array of results where the keys are the image style IDs
   * found and the values are the paths within the settings array where
   * these image styles are used.
   *
   * @param array $settings
   *   The settings array to be traversed.
   * @param array $ids
   *   An array of image style IDs to look for within the settings.
   * @param string $regex_pattern
   *   The regex pattern to match image style settings.
   *
   * @return array|bool
   *   An associative array where the keys are the image style IDs found and the
   *   values are the paths within the settings array where these image styles
   *   are used. Returns FALSE if no image styles are found.
   */
  protected function getUsagesInComponentSettings(array $settings, array $ids, string $regex_pattern): array|bool {
    $results = [];
    $path_helper = '';

    \array_walk_recursive($settings, function (mixed $value, string $key) use (&$results, $path_helper, $ids, $regex_pattern) {

      if (\in_array($value, $ids, TRUE) && preg_match("#$regex_pattern#", $key)) {
        $results[$value] = $path_helper . $key;
      }

      if (\is_array($value)) {
        $path_helper .= $key . '||';
      }
    });

    return !empty($results) ? $results : FALSE;
  }

  /**
   * Generates a link to the display mode configuration page for an entity.
   *
   * This method creates a link to the entity view display configuration page
   * for a given view mode, entity type, and bundle.
   *
   * @param string $view_mode
   *   The view mode of the entity (e.g., 'default', 'teaser').
   * @param string $entity_type_id
   *   The entity type ID (e.g., 'node', 'user').
   * @param string $bundle
   *   The bundle of the entity (e.g., 'article', 'page').
   *
   * @return \Drupal\Core\Link|null
   *   Link to the display mode configuration page, NULL otherwise.
   */
  protected function generateDisplayModeLink(string $view_mode, string $entity_type_id, string $bundle): ?Link {
    // Display modes link routes are add it
    // \Drupal\field_ui\Routing\RouteSubscriber::alterRoutes.
    $entity_definition = $this->entityTypeManager->getDefinition($entity_type_id);
    $link = NULL;

    if ($entity_definition instanceof EntityTypeInterface) {
      $parameters = FieldUI::getRouteBundleParameter($entity_definition, $bundle);

      if ($view_mode === 'default') {
        $route_name = "entity.entity_view_display.{$entity_type_id}.default";
      }
      else {
        $route_name = "entity.entity_view_display.{$entity_type_id}.view_mode";
        $parameters['view_mode_name'] = $view_mode;
      }

      $link = Link::createFromRoute($view_mode, $route_name, $parameters, [
        'attributes' => ['target' => '_blank'],
      ]);
    }

    return $link;
  }

}
