<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentModel;

use Drupal\Component\Utility\Html;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Drupal\xray_audit\Utils\XrayAuditTableFilter;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin entity field architecture.
 *
 * @XrayAuditTaskPlugin (
 *   id = "entity_architecture",
 *   label = @Translation("Entity architecture"),
 *   description = @Translation("Entity architecture."),
 *   group = "content_model",
 *   sort = 5,
 *   operations = {
 *      "content_entity_definition" = {
 *          "label" = "Content entity definitions",
 *          "description" = "Definitions of all content entities (nodes, paragraphs, blocks, etc.).",
 *          "download" = TRUE
 *       }
 *   },
 *   dependencies = {"field"},
 * )
 */
class XrayAuditEntityArchitecturePlugin extends XrayAuditTaskPluginBase {

  /**
   * Service "xray_audit.entity_field_architecture".
   *
   * @var \Drupal\xray_audit\Services\EntityArchitectureInterface
   */
  protected $entityArchitectureService;

  /**
   * Service "xray_audit.plugin_repository".
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityArchitectureService = $container->get('xray_audit.entity_architecture');
    $instance->pluginRepository = $container->get('xray_audit.plugin_repository');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    return [
      $this->t('Entity'),
      $this->t('Bundle'),
      $this->t('Type'),
      $this->t('Machine name'),
      $this->t('Label'),
      $this->t('Description'),
      $this->t('Data Type'),
      $this->t('Computed'),
      $this->t('Data Type Settings'),
      $this->t('Cardinality'),
      $this->t('Mandatory'),
      $this->t('Read Only'),
      $this->t('Translatable'),
      $this->t('Revisionable'),
      $this->t('Default value'),
      $this->t('Default value callback'),
      $this->t('Form widget'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    switch ($operation) {
      case 'content_entity_definition':
        return $this->entityArchitectureService->getDataForEntityFieldArchitecture();
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function buildDataRenderArray(array $data, string $operation = '') {
    $build = [];
    $rows = [];
    $headers = $this->getHeaders();

    foreach ($data as $data_row) {
      $rows[] = [
        "data" => [
          $data_row['content']['entity'] ?? '',
          $data_row['content']['bundle'] ?? '',
          $data_row['content']['type'] ?? '',
          $data_row['content']['machine_name'] ?? '',
          $data_row['content']['label'] ?? '',
          $data_row['content']['description'] ?? '',
          $data_row['content']['data_type'] ?? '',
          $data_row['content']['computed'] ?? '',
          $data_row['content']['data_type_settings'] ?? '',
          $data_row['content']['cardinality'] ?? '',
          $data_row['content']['mandatory'] ?? '',
          $data_row['content']['read_only'] ?? '',
          $data_row['content']['translatable'] ?? '',
          $data_row['content']['revisionable'] ?? '',
          $data_row['content']['default_value'] ?? '',
          $data_row['content']['default_value_callback'] ?? '',
          $data_row['content']['form_widget'] ?? '',
        ],
        "class" => ($data_row['content']['type'] === 'entity') ? ['xray-audit--highlighted'] : [],
      ];

    }

    $unique_id = Html::getUniqueId('xray-audit-table');

    $build['table'] = [
      '#type' => 'table',
      '#id' => $unique_id,
      '#header' => $headers,
      '#rows' => $rows,
      '#sticky' => TRUE,
      '#weight' => 10,
      '#attributes' => [
        'class' => ['xray-audit__table'],
      ],
      '#attached' => [
        'library' => [
          'xray_audit/xray_audit',
        ],
      ],
    ];

    // Process CSV download using the standardized base class method.
    $this->processCsvDownload($operation, $data, $build);

    $columns_indexes = [0, 1, 2, 3, 4, 6];
    $build['filter'] = XrayAuditTableFilter::generateRenderableFilterInput($unique_id, $columns_indexes, NULL, $headers);
    $build['filter']['#weight'] = 6;

    return $build;

  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvData(string $operation, array $data): array {
    $csv_rows = [];

    // Use the passed data if available, otherwise fetch it.
    $raw_data_rows = !empty($data) ? $data : $this->getDataOperationResult($operation);

    foreach ($raw_data_rows as $data_row) {
      $content = $data_row['content'] ?? [];
      $csv_row = [
        (string) ($content['entity'] ?? ''),
        (string) ($content['bundle'] ?? ''),
        (string) ($content['type'] ?? ''),
        (string) ($content['machine_name'] ?? ''),
        (string) ($content['label'] ?? ''),
        (string) ($content['description'] ?? ''),
        (string) ($content['data_type'] ?? ''),
        is_bool($content['computed'] ?? NULL) ? (($content['computed']) ? 'Yes' : 'No') : (string) ($content['computed'] ?? ''),
        (string) ($content['data_type_settings'] ?? ''),
        (string) ($content['cardinality'] ?? ''),
        is_bool($content['mandatory'] ?? NULL) ? (($content['mandatory']) ? 'Yes' : 'No') : (string) ($content['mandatory'] ?? ''),
        is_bool($content['read_only'] ?? NULL) ? (($content['read_only']) ? 'Yes' : 'No') : (string) ($content['read_only'] ?? ''),
        is_bool($content['translatable'] ?? NULL) ? (($content['translatable']) ? 'Yes' : 'No') : (string) ($content['translatable'] ?? ''),
        is_bool($content['revisionable'] ?? NULL) ? (($content['revisionable']) ? 'Yes' : 'No') : (string) ($content['revisionable'] ?? ''),
        (string) ($content['default_value'] ?? ''),
        (string) ($content['default_value_callback'] ?? ''),
        (string) ($content['form_widget'] ?? ''),
      ];
      $csv_rows[] = $csv_row;
    }

    return $csv_rows;
  }

}
