<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\Layout;

use Drupal\Core\Condition\ConditionPluginCollection;
use Drupal\Core\Link;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Drupal\xray_audit\XrayAuditTaskCsvDownloadTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of Block layout.
 *
 * @XrayAuditTaskPlugin (
 *   id = "block_layout",
 *   label = @Translation("Block layout"),
 *   description = @Translation("Block layout."),
 *   group = "layout",
 *   sort = 1,
 *   operations = {
 *      "block_configuration" = {
 *          "label" = "Block configurations",
 *          "download" = TRUE
 *      }
 *   },
 *   dependencies = {"block"}
 * )
 */
class XrayAuditBlockLayoutPlugin extends XrayAuditTaskPluginBase {

  use XrayAuditTaskCsvDownloadTrait;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    $instance->renderer = $container->get('renderer');

    return $instance;
  }

  /**
   * Formats visibility conditions into a list render array.
   *
   * @param \Drupal\Core\Condition\ConditionPluginCollection $conditions
   *   A collection of visibility conditions from the block config.
   *
   * @return array
   *   A list render array of visibility conditions.
   */
  protected function formatVisibilityConditions(ConditionPluginCollection $conditions): array {
    $list = [
      '#theme' => 'item_list',
      '#items' => [],
      '#attributes' => ['class' => ['visibility-conditions-list']],
    ];

    foreach ($conditions as $condition) {
      // Catch every exception that can be thrown by the condition plugin.
      // This way we can still display the other conditions.
      // And we prevent page from not being displayed.
      // A bad implementation in the summary method, for example using invalid
      // contexts, can cause a fatal error.
      try {
        $summary = (string) $condition->summary();
      }
      catch (\Exception $e) {
        $summary = $this->t('Unknown conditions');
      }
      finally {
        $plugin_definition = $condition->getPluginDefinition();
        $label = $plugin_definition['label'] ?? '';
        $list['#items'][] = $label . ': ' . $summary;
      }
    }

    if (empty($list['#items'])) {
      $list['#items'][] = $this->t('None');
    }

    return $list;
  }

  /**
   * Get condition summaries.
   *
   * @param \Drupal\Core\Condition\ConditionPluginCollection $conditions
   *   A collection of visibility conditions from the block config.
   *
   * @return array
   *   Condition summaries.
   */
  protected function getConditionSummaries(ConditionPluginCollection $conditions): array {
    $result = [];
    foreach ($conditions as $condition_id => $condition) {
      // Catch every exception that can be thrown by the condition plugin.
      // This way we can still display the other conditions.
      // And we prevent page from not being displayed.
      // A bad implementation in the summary method, for example using invalid
      // contexts, can cause a fatal error.
      try {
        $summary = (string) $condition->summary();
      }
      catch (\Exception $e) {
        $summary = $this->t('Unknown conditions');
      }
      finally {
        $plugin_definition = $condition->getPluginDefinition();
        $label = $plugin_definition['label'] ?? '';
        $result[$condition_id] = $label . ': ' . $summary;
      }
    }
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    return [
      'id' => $this->t('Id'),
      'label' => $this->t('Label'),
      'theme' => $this->t('Theme'),
      'region' => $this->t('Region'),
      'status' => $this->t('Status'),
      'visibility_conditions' => $this->t('Visibility Conditions'),
      'link' => $this->t('Link'),
    ];

  }

  /**
   * Gets the rows for the 'block_configuration' operation.
   *
   * Returns both table rows for display and raw visibility data for CSV.
   *
   * @return array
   *   An associative array containing 'table_rows' and 'csv_visibility_data'.
   */
  private function getBlockConfigurationRows(): array {
    $system_theme_config = $this->configFactory->get('system.theme');
    $default_theme = $system_theme_config->get('default');
    $admin_theme = $system_theme_config->get('admin');

    $block_storage = $this->entityTypeManager->getStorage('block');
    /** @var \Drupal\block\Entity\Block[] $blocks */
    $blocks = $block_storage->loadMultiple();
    $table_rows = [];

    foreach ($blocks as $block) {
      $plane_visibility_conditions = $this->getConditionSummaries($block->getVisibilityConditions());
      $formatted_visibility_conditions = [
        '#theme' => 'item_list',
        '#items' => $plane_visibility_conditions,
        '#attributes' => ['class' => ['visibility-conditions-list']],
      ];
      $theme = $block->getTheme();
      $theme_label = $theme;
      if ($theme == $default_theme) {
        $theme_label = $theme . ' (default)';
      }
      elseif ($theme == $admin_theme) {
        $theme_label = $theme . ' (admin)';
      }

      $table_rows[$block->id()] = [
        'id' => $block->id(),
        'title' => $block->label(),
        'theme' => $theme_label,
        'region' => $block->getRegion(),
        'status' => $block->status() ? $this->t('Enabled') : $this->t('Disabled'),
        'visibility' => $this->renderer->render($formatted_visibility_conditions),
        'link' => Link::fromTextAndUrl($this->t('Block page'), $block->toUrl()),
        'plane_visibility' => $plane_visibility_conditions,
      ];
    }

    array_multisort(
      array_column($table_rows, 'theme'),
      array_column($table_rows, 'region'),
      $table_rows
    );

    return $table_rows;
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    return $this->getBlockConfigurationRows();
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {

    $cid = $this->getPluginId() . ':' . $operation;
    // Using config:block.block.* and system.theme as cache tags.
    $cache_tags = $this->entityTypeManager->getDefinition('block')->getListCacheTags();
    $cache_tags = array_merge($cache_tags, ['config:system.theme']);

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $headers = $this->getHeaders($operation);
    $rowData = $this->getRows($operation);

    $results_table = $rowData;

    $data = [
      'header_table' => $headers,
      'results_table' => $results_table,
    ];

    $this->pluginRepository->setCacheTagsInv($cid, $data, $cache_tags);
    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvHeaders(string $operation): array {
    $headers = $this->getHeaders($operation);
    unset($headers['link']);
    return $headers;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvData(string $operation, array $data): array {
    foreach ($data as &$item) {
      unset($item['link']);
      $item['visibility'] = implode(', ', $item['plane_visibility']);
      unset($item['plane_visibility']);
      $item['status'] = $item['status'] == $this->t('Enabled') ? '1' : '0';
    }
    return $data;
  }

}
