<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\Packages;

use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of themes status.
 *
 * @XrayAuditTaskPlugin (
 *   id = "themes",
 *   label = @Translation("Themes Status (sync config files)"),
 *   description = @Translation("Themes status against sync configuration files (include config splits)."),
 *   group = "package",
 *   sort = 1,
 *   operations = {
 *      "themes" = {
 *          "label" = "Themes",
 *          "description" = "Installed themes and their configuration status",
 *          "download" = TRUE
 *       }
 *    },
 * )
 */
class XrayAuditThemesPlugin extends XrayAuditTaskPluginBase {

  /**
   * Service "theme_handler".
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * Service "xray_audit.plugin_repository".
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Service "config.factory".
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->themeHandler = $container->get('theme_handler');
    $instance->pluginRepository = $container->get('xray_audit.plugin_repository');
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    switch ($operation) {

      case 'themes':
        return $this->getThemeStatus();

    }
    return [];

  }

  /**
   * Modules custom and contrib.
   *
   * @return array
   *   Data.
   */
  protected function getThemeStatus() {
    $resultTable = [];
    $list_themes_installed = $this->themeHandler->listInfo();
    if (empty($list_themes_installed)) {
      return $resultTable;
    }
    $default_admin_theme = $this->configFactory->get('system.theme')->get('admin');
    $default_admin_theme_name = $this->themeHandler->getName($default_admin_theme);
    // Build the data.
    foreach ($list_themes_installed as $machine_name => $theme) {
      $info = $theme->info;
      $group = (isset($info['package']) && stristr($info['package'], 'core')) ? 'core' : 'not_core';
      $resultTable[$group][$machine_name] = [
        'machine_name' => $machine_name,
        'project' => $info['project'] ?? '',
        'Theme' => $info['name'],
        'Default' => $info['name'] == $this->themeHandler->getDefault() || $info['name'] == $default_admin_theme_name ? $this->t('Yes') : $this->t('No'),
        'version' => $info['version'] ?? '-',
      ];
    }

    return $resultTable;

  }

  /**
   * {@inheritDoc}
   */
  public function buildDataRenderArray(array $data, string $operation = '') {
    $group_info = [];

    if (empty($data)) {
      return [
        "#markup" => $this->t("Not data found,
              maybe the configuration files have not been exported."),
      ];
    }

    $group_info['core'] = [
      'title' => $this->t("Core Themes"),
      'description' => $this->t("Themes belong to the core"),
    ];
    $group_info['not_core'] = [
      'title' => $this->t("Themes custom and contrib"),
      'description' => $this->t("Themes custom and contrib"),
    ];

    $builds = [];
    foreach ($data as $type => $group_themes) {
      // Sort the modules by package.
      $key_values = array_column($group_themes, 'Theme');
      array_multisort($key_values, SORT_ASC, $group_themes);

      $builds[$type] = [
        '#theme' => 'details',
        '#title' => $group_info[$type]['title'],
        '#description' => $group_info[$type]['description'],
        '#attributes' => ['class' => ['package-listing']],
        '#summary_attributes' => [],
        '#children' => [
          'tables' => [
            '#theme' => 'table',
            '#header' => $this->getHeaders($operation),
            '#rows' => $group_themes,
          ],
        ],
      ];
    }

    // Process CSV download using the standardized base class method.
    $this->processCsvDownload($operation, $data, $builds);

    return $builds;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    // Headers for the HTML table.
    return [
      $this->t('Machine name'),
      $this->t('Project'),
      $this->t('Theme'),
      $this->t('Used'),
      $this->t('Version'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvHeaders(string $operation = ''): array {
    // Headers for the CSV file.
    return [
      $this->t('Project')->render(),
      $this->t('Theme Name')->render(),
      $this->t('Machine Name')->render(),
      $this->t('Used (Default/Admin)')->render(),
      $this->t('Version')->render(),
      $this->t('Type (Core/Contrib-Custom)')->render(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvData(string $operation, array $data): array {
    $csv_rows = [];

    // Use the passed data if available, otherwise fetch it.
    $theme_data_grouped = !empty($data) ? $data : $this->getThemeStatus();

    if (empty($theme_data_grouped)) {
      return [];
    }

    foreach ($theme_data_grouped as $group_name => $themes_in_group) {
      foreach ($themes_in_group as $theme_info) {
        $csv_rows[] = [
          (string) ($theme_info['project'] ?? ''),
          (string) ($theme_info['Theme'] ?? ''),
          (string) ($theme_info['machine_name'] ?? ''),
          (string) ($theme_info['Default'] ?? $this->t('No')),
          (string) ($theme_info['version'] ?? '-'),
          (string) ($theme_info['group'] ?? $group_name),
        ];
      }
    }
    return $csv_rows;
  }

}
