<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\SiteStructure;

use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\xray_audit\Form\MenuSelectorForm;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "navigation_architecture",
 *   label = @Translation("Navigation Architecture"),
 *   description = @Translation("Navigation Architecture."),
 *   group = "site_structure",
 *   sort = 1,
 *   operations = {
 *      "menu" = {
 *          "label" = "Menus",
 *          "description" = "Menu structures.",
 *          "dependencies" = {"menu_link_content"},
 *          "download" = TRUE,
 *       }
 *   }
 * )
 */
final class XrayAuditNavigationArchitecturePlugin extends XrayAuditTaskPluginBase {

  /**
   * Service "navigation_architecture".
   *
   * @var \Drupal\xray_audit\Services\NavigationArchitectureInterface
   */
  protected $navigationArchitecture;

  /**
   * Service "xray_audit.plugin_repository".
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Service "request_stack".
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Service "form_builder".
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->navigationArchitecture = $container->get('xray_audit.navigation_architecture');
    $instance->pluginRepository = $container->get('xray_audit.plugin_repository');
    $instance->requestStack = $container->get('request_stack');
    $instance->formBuilder = $container->get('form_builder');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    switch ($operation) {
      case 'menu':
        $query_parameters_values = $this->getQueryParametersFromRequest();
        $show_parent_reference = !empty($query_parameters_values['show_parent_reference']);

        // Get the requested menu or default to 'main'.
        $menu_id = $query_parameters_values['menu'] ?? 'main';

        // Handle "all" menus selection.
        if ($menu_id === 'all') {
          return $this->getAllMenusData($show_parent_reference);
        }

        // Get the architecture for the specific menu.
        $menu_architecture = $this->navigationArchitecture->getMenuArchitecture($menu_id);

        return [
          'menu' => $menu_id,
          'items' => $menu_architecture['items'] ?? [],
          'item_number' => $menu_architecture['item_number'] ?? 0,
          'level_max' => $menu_architecture['level_max'] ?? 0,
          'show_parent_reference' => $show_parent_reference,
        ];
    }

    return [];
  }

  /**
   * Get data for all available menus.
   *
   * @param bool $show_parent_reference
   *   Whether to show parent references in level columns.
   *
   * @return array
   *   Combined data for all menus.
   */
  protected function getAllMenusData(bool $show_parent_reference): array {
    $menu_list = $this->navigationArchitecture->getMenuList();
    $all_items = [];
    $total_items = 0;
    $max_level = 0;

    // Loop through each menu and get its architecture.
    foreach ($menu_list as $menu_id => $menu_label) {
      $menu_architecture = $this->navigationArchitecture->getMenuArchitecture($menu_id);

      if (!empty($menu_architecture['items'])) {
        // Add menu information to each item.
        foreach ($menu_architecture['items'] as $item) {
          $item['menu_id'] = $menu_id;
          $item['menu_label'] = $menu_label;
          $all_items[] = $item;
        }
        $total_items += $menu_architecture['item_number'] ?? 0;

        // Track the maximum level across all menus.
        if (isset($menu_architecture['level_max']) && $menu_architecture['level_max'] > $max_level) {
          $max_level = $menu_architecture['level_max'];
        }
      }
    }

    return [
      'menu' => 'all',
      'items' => $all_items,
      'item_number' => $total_items,
      'level_max' => $max_level,
      'show_parent_reference' => $show_parent_reference,
    ];
  }

  /**
   * Get data from query parameters.
   *
   * @return mixed[]
   *   Data from query parameters.
   */
  protected function getQueryParametersFromRequest(): array {
    $data = ['menu' => NULL, 'show_parent_reference' => NULL];
    $key_items = array_keys($data);

    /**@var \Symfony\Component\HttpFoundation\Request $request*/
    $request = $this->requestStack->getCurrentRequest();
    if (!$request instanceof Request) {
      return $data;
    }

    foreach ($key_items as $item_key) {
      $data[$item_key] = $request->query->get($item_key, NULL);
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   *
   * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
   * @SuppressWarnings(PHPMD.NPathComplexity)
   * @SuppressWarnings(PHPMD.CyclomaticComplexity)
   */
  public function buildDataRenderArray(array $data, string $operation = '') {
    $build = [];

    // Add menu selector form.
    // The form allows users to select different menus and toggle
    // parent reference display.
    $build['form'] = $this->formBuilder->getForm(MenuSelectorForm::class);
    $build['form']['#weight'] = 10;

    if (empty($data['items'])) {
      return $build;
    }

    $menu_id = $data['menu'];

    // Get human-readable menu name from NavigationArchitecture service.
    if ($menu_id === 'all') {
      $menu_label = $this->t('All menus');
    }
    else {
      $menu_list = $this->navigationArchitecture->getMenuList();
      $menu_label = $menu_list[$menu_id] ?? $menu_id;
    }

    $headers = [];

    // Add menu as first column.
    $headers['menu'] = $this->t('Menu');

    // Build dynamic headers based on max level.
    for ($i = 1; $i <= $data['level_max']; $i++) {
      $headers['l_' . $i] = $this->t('L@level', ['@level' => $i]);
    }

    $headers['level'] = $this->t('Level');
    $headers['enabled'] = $this->t('Enabled');
    $headers['link'] = $this->t('Link');
    $headers['target'] = $this->t('Link description');
    $headers['operation'] = $this->t('Operations');

    $rows = [];

    // Prepare rows for HTML table.
    foreach ($data['items'] as $item) {
      $row = [];

      // Add menu name as first column.
      // When viewing all menus, use the specific menu label for each item.
      if ($menu_id === 'all' && isset($item['menu_label'])) {
        $row['menu'] = $item['menu_label'];
      }
      else {
        $row['menu'] = $menu_label;
      }

      for ($i = 1; $i <= $data['level_max']; $i++) {
        switch (TRUE) {
          case $i < $item['level'] && isset($item['levels'][$i]):
            // Show parent level title when show_parent_reference is enabled.
            $parent_title = '';
            if ($data['show_parent_reference']) {
              // Show just the title at this specific level.
              $parent_title = $item['levels'][$i];
            }
            $row['l_' . $i] = [
              'data' => $parent_title,
              'class' => ['xray-audit-cell-background-color-gray'],
            ];
            break;

          case $i === $item['level']:
            $row['l_' . $i] = $item['title'];
            break;

          default:
            $row['l_' . $i] = '';
        }
      }

      $row['level'] = $item['level'];
      $row['enabled'] = ($item['enabled']) ?
        $this->t('Yes') : $this->t('No');
      $row['link'] = !empty($item['link']) ?
        Link::fromTextAndUrl($item['link']->toString(), $item['link']) : '';
      $row['target'] = empty($item['target']) ?
        '' : implode(' ', $item['target']);

      // Add edit operation link.
      // Use the edit_link provided by NavigationArchitecture service.
      // This handles all menu link types correctly
      // (content entities, system links, etc).
      // System/plugin-based links may not have edit routes
      // and will have empty edit_link.
      if (!empty($item['edit_link']) && $item['edit_link'] instanceof Url) {
        $row['operation'] = Link::fromTextAndUrl($this->t('Edit'), $item['edit_link']);
      }
      else {
        // No edit link available (e.g., system links, external links).
        $row['operation'] = '';
      }

      $rows[] = $row;
    }

    $build['main_table'] = [
      '#theme' => 'table',
      '#header' => $headers,
      '#rows' => $rows,
      '#weight' => 30,
      '#responsive' => TRUE,
      '#sticky' => TRUE,
      '#attributes' => [
        'class' => [
          'xray-audit__table',
          'alignment-vertical-center',
        ],
      ],
      '#attached' => [
        'library' => [
          'xray_audit/xray_audit',
        ],
      ],
    ];

    // Process CSV download using the standardized base class method.
    $this->processCsvDownload($operation, $data, $build);

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    $headers = [];
    // Add menu as first column.
    $headers['menu'] = $this->t('Menu');
    $headers['level'] = $this->t('Level');
    $headers['enabled'] = $this->t('Enabled');
    $headers['link'] = $this->t('Link');
    $headers['target'] = $this->t('Link description');
    $headers['operation'] = $this->t('Operations');
    return $headers;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvHeaders(string $operation = ''): array {
    $data = $this->getDataOperationResult($operation);
    $csv_headers = [];

    // Add menu as first column.
    $csv_headers[] = $this->t('Menu')->render();

    if (isset($data['level_max'])) {
      for ($i = 1; $i <= $data['level_max']; $i++) {
        $csv_headers[] = $this->t('L@level', ['@level' => $i])->render();
      }
    }

    $csv_headers[] = $this->t('Level')->render();
    $csv_headers[] = $this->t('Enabled')->render();
    $csv_headers[] = $this->t('Link URL')->render();
    $csv_headers[] = $this->t('Link Title')->render();
    $csv_headers[] = $this->t('Link Description')->render();
    return $csv_headers;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvData(string $operation, array $data): array {
    $csv_rows = [];

    if (empty($data['items'])) {
      return [];
    }

    // Get human-readable menu name from NavigationArchitecture service.
    $menu_id = $data['menu'];
    if ($menu_id === 'all') {
      $menu_label = $this->t('All menus');
    }
    else {
      $menu_list = $this->navigationArchitecture->getMenuList();
      $menu_label = $menu_list[$menu_id] ?? $menu_id;
    }

    foreach ($data['items'] as $item) {
      $row = [];

      // Add menu name as first column.
      // When viewing all menus, use the specific menu label for each item.
      if ($menu_id === 'all' && isset($item['menu_label'])) {
        $row[] = (string) $item['menu_label'];
      }
      else {
        $row[] = (string) $menu_label;
      }

      for ($i = 1; $i <= $data['level_max']; $i++) {
        if ($i < $item['level'] && isset($item['levels'][$i])) {
          // Show parent level title when show_parent_reference is enabled.
          $parent_title = '';
          if ($data['show_parent_reference']) {
            // Show just the title at this specific level.
            $parent_title = $item['levels'][$i];
          }
          $row[] = (string) $parent_title;
        }
        elseif ($i === $item['level']) {
          $row[] = (string) $item['title'];
        }
        else {
          $row[] = '';
        }
      }

      $row[] = (string) $item['level'];
      $row[] = ($item['enabled']) ? $this->t('Yes')->render() : $this->t('No')->render();
      $row[] = !empty($item['link']) && $item['link'] instanceof Url ? $item['link']->toString() : (!empty($item['link']) && is_string($item['link']) ? $item['link'] : '');
      $row[] = !empty($item['link']) && $item['link'] instanceof Link ? $item['link']->getText() : ((string) ($item['title'] ?? ''));
      $row[] = empty($item['target']) ? '' : implode(' ', $item['target']);

      $csv_rows[] = $row;
    }
    return $csv_rows;
  }

}
