<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\SiteStructure;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Link;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\Core\Url;
use Drupal\views\Plugin\views\cache\CachePluginBase;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\ViewEntityInterface;
use Drupal\views\ViewExecutable;
use Drupal\views\Views;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "views",
 *   label = @Translation("Views"),
 *   description = @Translation("Views."),
 *   group = "site_structure",
 *   sort = 2,
 *   operations = {
 *      "views" = {
 *          "label" = "Views",
 *          "description" = "Enabled views and cache configuration.",
 *          "dependencies" = {"views"},
 *          "download" = TRUE
 *       }
 *   }
 * )
 */
final class XrayAuditViewsPlugin extends XrayAuditTaskPluginBase {

  /**
   * Messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Route provider.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->messenger = $container->get('messenger');
    $instance->routeProvider = $container->get('router.route_provider');
    return $instance;

  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    $headers = [
      $this->t('Module'),
      $this->t('View ID'),
      $this->t('Display ID'),
      $this->t('View label'),
      $this->t('Display label'),
      $this->t('Plugin display'),
      $this->t('Cache plugin'),
      $this->t('Cache duration'),
      $this->t('Cache Tags'),
      $this->t('Path'),
      $this->t('Access Configuration'),
      $this->t('Accessible for anonymous user'),
    ];

    if ($this->moduleHandler->moduleExists('views_ui')) {
      $headers[] = $this->t('Operation');
    }

    return $headers;
  }

  /**
   * Gets the rows for the 'views' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getViewsRows(): array {
    $rows = [];
    $view_storage = $this->entityTypeManager->getStorage('view');
    $views = $view_storage->loadMultiple();
    $definition = $this->getPluginDefinition();
    $destination_url = $definition['operations']['views']['url'] ?? '';

    foreach ($views as $view) {
      if (!($view instanceof ViewEntityInterface && $view->status())) {
        continue;
      }

      $view_id = $view->id();
      if (empty($view_id)) {
        continue;
      }

      $executable_view = Views::getView((string) $view_id);
      if (!$executable_view instanceof ViewExecutable) {
        continue;
      }
      $displays = $executable_view->storage->get('display');
      foreach ($displays as $display) {
        $rowData = $this->getDisplayData($view, $executable_view, $display);

        if ($this->moduleHandler->moduleExists('views_ui')) {
          $edit_url = Url::fromRoute('entity.view.edit_display_form', [
            'view' => $rowData['id_view'],
            'display_id' => $rowData['id_display'],
          ], [
            'query' => ['destination' => $destination_url],
          ]);

          if ($edit_url->access()) {
            $rowData['edit_link'] = Link::fromTextAndUrl($this->t('Edit'), $edit_url);
          }
          else {
            $rowData['edit_link'] = $this->t('No access to edit');
          }
        }
        $rows[] = $rowData;
      }
    }

    // Sort by module and view id.
    usort($rows, function ($a, $b) {
      if ($a['module'] === $b['module']) {
        return $a['id_view'] <=> $b['id_view'];
      }
      return $a['module'] <=> $b['module'];
    });
    return $rows;
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    return $this->getViewsRows();
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {

    $cid = $this->getPluginId() . ':' . $operation;
    $cache_tags = ['views_list', 'config:views.view.*'];

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $data = [
      'header_table' => $this->getHeaders($operation),
      'results_table' => $this->getRows($operation),
    ];

    $this->pluginRepository->setCacheTagsInv($cid, $data, $cache_tags);
    return $data;
  }

  /**
   * Get display data.
   *
   * @param \Drupal\views\ViewEntityInterface $view
   *   View object.   *.
   * @param \Drupal\views\ViewExecutable $viewExecutable
   *   Display object.   *.
   * @param array $display
   *   Display data.
   *
   * @return array
   *   Display data.
   */
  protected function getDisplayData(ViewEntityInterface $view, ViewExecutable $viewExecutable, array $display): array {

    $data = [
      'module' => $view->get('module'),
      'id_view' => $viewExecutable->id(),
      'id_display' => $display['id'],
      'label_view' => $view->label(),
      'label_display' => $display['display_title'],
      'plugin_display' => $display['display_plugin'],
      'cache_plugin_id' => '',
      'cache_max_age' => '',
      'cache_tags' => '',
      'path' => '',
      'access' => '',
      'anonymous_user_can_access' => FALSE,
      'warning_anonymous_user_can_access' => FALSE,
    ];

    $is_admin_route = FALSE;

    $viewExecutable->setDisplay($display['id']);
    $display_object = $viewExecutable->getDisplay();

    if (!$display_object instanceof DisplayPluginBase) {
      return $data;
    }

    $path_info = $this->pathFromRouteNameAndIfItIsAdmin('view.' . $data['id_view'] . '.' . $data['id_display']);
    if ($path_info) {
      $data['path'] = $path_info['path'];
      $is_admin_route = $path_info['isAdminRoute'];
    }

    // Access.
    if ($data['plugin_display'] !== 'default') {
      $access = $display_object->getOption('access');
      if (is_array($access)) {
        $flat_access = new \RecursiveIteratorIterator(new \RecursiveArrayIterator($access));
        $data['access'] = implode(' | ', iterator_to_array($flat_access));
      }

      // Anonymous user access.
      $data['anonymous_user_can_access'] = $display_object->access(new AnonymousUserSession());

      // Check if it could be a security issue that anonymous user can access.
      $views_could_be_accessible_by_anonymous = $this->configFactory->get('xray_audit.views_report')->get('admin_views_anonymous') ?? [];
      if ($data['anonymous_user_can_access'] &&
        $is_admin_route &&
        !in_array($data['id_view'] . '.' . $data['id_display'], $views_could_be_accessible_by_anonymous)) {
        $data['warning_anonymous_user_can_access'] = TRUE;
      }

    }

    // Cache.
    $cache_plugin = $display_object->getPlugin('cache');
    if ($cache_plugin instanceof CachePluginBase) {
      $cache_max_age = $cache_plugin->getCacheMaxAge();

      switch ($cache_max_age) {
        case 0:
          $cache_max_age = 'No cache';
          break;

        case -1:
          $cache_max_age = 'Cache permanent';
          break;

        default:
          $cache_max_age = $cache_max_age . ' seconds';
          break;
      }

      $data['cache_plugin_id'] = $cache_plugin->getPluginId();
      $data['cache_max_age'] = $cache_max_age;
      $data['cache_tags'] = implode(', ', $cache_plugin->getCacheTags());
    }

    return $data;

  }

  /**
   * Get path from route name and if it is admin.
   *
   * @param string $routeName
   *   Route name.
   *
   * @return array|null
   *   Path and if it is admin.
   */
  protected function pathFromRouteNameAndIfItIsAdmin(string $routeName): ?array {
    $result = ['path' => '', 'isAdminRoute' => FALSE];
    $routes = $this->routeProvider->getRoutesByNames([$routeName]);
    if (empty($routes)) {
      return NULL;
    }
    $route = reset($routes);
    $result['path'] = $route->getPath();
    $options = $route->getOptions() ?? [];
    $result['isAdminRoute'] = !empty($options['_admin_route']);
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function buildDataRenderArray(array $data, string $operation = ''): array {

    $this->preprocessAccessAnonymousUser($data['results_table']);

    // The 'Operation' column and its data are now handled in getHeaders() and getViewsRows().
    // We still need to display the warning if views_ui is not enabled.
    if (!$this->moduleHandler->moduleExists('views_ui')) {
      $this->messenger->addWarning(
        $this->t('If you want to have the Operations column and access view editing, please enable the Views UI module.'));
    }

    $build = parent::buildDataRenderArray($data, $operation);

    return $build;
  }

  /**
   * Preprocess anonymous user access data.
   *
   * @param array $rows
   *   Rows.
   */
  protected function preprocessAccessAnonymousUser(array &$rows) {
    foreach ($rows as &$row) {
      $inline_style = '';
      if ($row['warning_anonymous_user_can_access']) {
        $inline_style = 'color: red';
      }
      $row['anonymous_user_can_access'] = new FormattableMarkup(
        '<span style="@inline_class";>@txt</span>',
        [
          '@inline_class' => @$inline_style,
          '@txt' => $row['anonymous_user_can_access'] ? 'Yes' : 'No',
        ]
      );
      unset($row['warning_anonymous_user_can_access']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvHeaders(string $operation): array {
    $headers = $this->getHeaders($operation);
    $operation_header_text = (string) $this->t('Operation');

    $headers = array_filter($headers, function ($header_item) use ($operation_header_text) {
        return (string) $header_item !== $operation_header_text;
    });

    return array_values($headers);
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvData(string $operation, array $data): array {

    foreach ($data as &$row) {
      // 'edit_link' might or might not exist based on views_ui.
      // Unset it if it exists to prevent it from being in CSV.
      if (isset($row['edit_link'])) {
        unset($row['edit_link']);
      }
      if (isset($row['anonymous_user_can_access'])) {
        $row['anonymous_user_can_access'] = $row['anonymous_user_can_access']->__toString() == 'Yes' ? 1 : 0;
      }
    }
    return $data;
  }

}
