<?php

namespace Drupal\Tests\xray_audit\Functional\Controller;

use Drupal\xray_audit\Controller\XrayAuditGroupsController;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the XrayAuditGroupsController.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 * @group critical
 */
class XrayAuditGroupsControllerTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'xray_audit',
  ];

  /**
   * A user with permission to access xray audit.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A user without permission to access xray audit.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * The plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * The group plugin manager service.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditGroupPluginManager
   */
  protected $groupPluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create users with different permissions.
    $this->privilegedUser = $this->drupalCreateUser([
      'xray_audit access',
      'access content',
    ]);

    $this->unprivilegedUser = $this->drupalCreateUser([
      'access content',
    ]);

    // Get services.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');
    $this->groupPluginManager = $this->container->get('plugin_manager.xray_audit_group');
  }

  /**
   * Tests that unprivileged users cannot access group routes.
   */
  public function testAccessDeniedForUnprivilegedUsers() {
    // Arrange: Login as unprivileged user.
    $this->drupalLogin($this->unprivilegedUser);

    // Get a group to test with.
    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($group_definitions, 'Should have group definitions');

    $first_group = reset($group_definitions);
    $group_id = str_replace('_', '-', $first_group['id']);

    // Act: Try to access group route.
    $this->drupalGet('/admin/reports/xray-audit/' . $group_id);

    // Assert: Access should be denied.
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that privileged users can access group routes.
   */
  public function testAccessGrantedForPrivilegedUsers() {
    // Arrange: Login as privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Get a group to test with.
    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($group_definitions, 'Should have group definitions');

    $first_group = reset($group_definitions);
    $group_id = str_replace('_', '-', $first_group['id']);

    // Act: Access group route.
    $this->drupalGet('/admin/reports/xray-audit/' . $group_id);

    // Assert: Should be accessible.
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests build() returns proper render array.
   */
  public function testBuildReturnsRenderArray() {
    // Arrange: Get controller and group ID.
    $controller = XrayAuditGroupsController::create($this->container);

    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($group_definitions, 'Should have group definitions');

    $first_group = reset($group_definitions);
    $group_id = $first_group['id'];

    // Act: Call build method.
    $render_array = $controller->build($group_id);

    // Assert: Should return an array.
    $this->assertIsArray($render_array, 'build() should return an array');
    $this->assertNotEmpty($render_array, 'build() should return non-empty array');
  }

  /**
   * Tests build() calls buildTaskItems for the group.
   */
  public function testBuildCallsBuildTaskItems() {
    // Arrange: Get controller and group ID.
    $controller = XrayAuditGroupsController::create($this->container);

    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($group_definitions, 'Should have group definitions');

    $first_group = reset($group_definitions);
    $group_id = $first_group['id'];

    // Act: Build render array.
    $render_array = $controller->build($group_id);

    // Assert: Should have content from buildTaskItems.
    $this->assertIsArray($render_array);

    // Find admin_block_content and check it has content.
    $found_content = FALSE;
    foreach ($render_array as $element) {
      if (isset($element['#theme']) && $element['#theme'] === 'admin_block_content') {
        $this->assertArrayHasKey('#content', $element);
        $found_content = TRUE;
        break;
      }
    }

    $this->assertTrue($found_content, 'Should have content from buildTaskItems');
  }

  /**
   * Tests getGroupTitle() returns correct title.
   */
  public function testGetGroupTitleReturnsCorrectTitle() {
    // Arrange: Get controller and group.
    $controller = XrayAuditGroupsController::create($this->container);

    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($group_definitions, 'Should have group definitions');

    $first_group = reset($group_definitions);
    $group_id = $first_group['id'];
    $expected_label = (string) $first_group['label'];

    // Act: Get group title.
    $title = $controller->getGroupTitle($group_id);

    // Assert: Should return the group label.
    $this->assertIsString($title, 'getGroupTitle() should return a string');
    $this->assertEquals($expected_label, $title, 'Title should match group label');
  }

  /**
   * Tests getGroupTitle() returns not found message for invalid group.
   */
  public function testGetGroupTitleReturnsNotFoundForInvalidGroup() {
    // Arrange: Get controller.
    $controller = XrayAuditGroupsController::create($this->container);

    // Act: Get title for invalid group.
    $title = $controller->getGroupTitle('invalid_group_that_does_not_exist');

    // Assert: Should return not found message.
    $this->assertIsString($title, 'getGroupTitle() should return a string');
    $this->assertStringContainsString('not found', $title, 'Title should indicate group not found');
  }

  /**
   * Tests group page displays correct content.
   */
  public function testGroupPageDisplaysContent() {
    // Arrange: Login as privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Get a group to test with.
    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($group_definitions, 'Should have group definitions');

    $first_group = reset($group_definitions);
    $group_id = str_replace('_', '-', $first_group['id']);

    // Act: Access group route.
    $this->drupalGet('/admin/reports/xray-audit/' . $group_id);

    // Assert: Page should display content.
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests group page title callback.
   */
  public function testGroupPageTitleCallback() {
    // Arrange: Login as privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Get a group to test with.
    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($group_definitions, 'Should have group definitions');

    $first_group = reset($group_definitions);
    $group_id = str_replace('_', '-', $first_group['id']);
    $expected_title = (string) $first_group['label'];

    // Act: Access group route.
    $this->drupalGet('/admin/reports/xray-audit/' . $group_id);

    // Assert: Page title should match group label.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains($expected_title);
  }

  /**
   * Tests that build renders task items from the specified group only.
   */
  public function testBuildRendersOnlySpecifiedGroup() {
    // Arrange: Get controller.
    $controller = XrayAuditGroupsController::create($this->container);

    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($group_definitions, 'Should have group definitions');

    $first_group = reset($group_definitions);
    $group_id = $first_group['id'];

    // Act: Build for specific group.
    $render_array = $controller->build($group_id);

    // Assert: Should render content for that group only.
    $this->assertIsArray($render_array);
    $this->assertNotEmpty($render_array, 'Should have content for the group');

    // Verify structure is correct for single group content.
    foreach ($render_array as $element) {
      if (isset($element['#theme']) && $element['#theme'] === 'admin_block_content') {
        $this->assertArrayHasKey('#content', $element);
        break;
      }
    }
  }

}
