<?php

namespace Drupal\Tests\xray_audit\Kernel\Form;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormState;
use Drupal\KernelTests\KernelTestBase;
use Drupal\xray_audit\Form\MenuSelectorForm;

/**
 * Tests the MenuSelectorForm.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class MenuSelectorFormTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'xray_audit',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\xray_audit\Form\MenuSelectorForm
   */
  protected $form;

  /**
   * The navigation architecture service.
   *
   * @var \Drupal\xray_audit\Services\NavigationArchitectureInterface
   */
  protected $navigationArchitecture;

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['system']);

    // Get services.
    $this->navigationArchitecture = $this->container->get('xray_audit.navigation_architecture');
    $this->formBuilder = $this->container->get('form_builder');

    // Create form instance.
    $this->form = MenuSelectorForm::create($this->container);
  }

  /**
   * Tests form instantiation through dependency injection.
   */
  public function testFormInstantiation() {
    // Arrange & Act: Get form from container.
    $form = MenuSelectorForm::create($this->container);

    // Assert: Form is instantiated correctly.
    $this->assertInstanceOf(
      MenuSelectorForm::class,
      $form,
      'Form should be properly instantiated'
    );
  }

  /**
   * Tests that form receives correct dependencies.
   */
  public function testFormDependencies() {
    // Arrange: Get request stack service.
    $request_stack = $this->container->get('request_stack');

    // Act: Create form with dependencies.
    $form = new MenuSelectorForm(
      $this->navigationArchitecture,
      $request_stack
    );

    // Assert: Form is created successfully.
    $this->assertInstanceOf(
      MenuSelectorForm::class,
      $form,
      'Form should accept dependencies in constructor'
    );
  }

  /**
   * Tests that form implements FormInterface.
   */
  public function testFormImplementsFormInterface() {
    // Assert: Form implements FormInterface.
    $this->assertInstanceOf(
      FormInterface::class,
      $this->form,
      'Form should implement FormInterface'
    );
  }

  /**
   * Tests getFormId returns the correct form ID.
   */
  public function testGetFormId() {
    // Act: Get form ID.
    $form_id = $this->form->getFormId();

    // Assert: Form ID is correct.
    $this->assertEquals(
      'xray_audit_menu_selector_form',
      $form_id,
      'Form ID should be xray_audit_menu_selector_form'
    );
  }

  /**
   * Tests that getFormId returns a string.
   */
  public function testGetFormIdReturnsString() {
    // Act: Get form ID.
    $form_id = $this->form->getFormId();

    // Assert: Form ID is a string.
    $this->assertIsString($form_id, 'Form ID should be a string');
    $this->assertNotEmpty($form_id, 'Form ID should not be empty');
  }

  /**
   * Tests that buildForm returns proper form array structure.
   */
  public function testBuildFormStructure() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Form has required elements.
    $this->assertArrayHasKey('menu', $form_array, 'Form should have menu field');
    $this->assertArrayHasKey('show_parent_reference', $form_array, 'Form should have show_parent_reference field');
    $this->assertArrayHasKey('actions', $form_array, 'Form should have actions');
  }

  /**
   * Tests that menu select field has correct structure.
   */
  public function testMenuSelectFieldStructure() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Menu field has correct properties.
    $this->assertEquals('select', $form_array['menu']['#type'], 'Menu field should be a select');
    $this->assertNotEmpty($form_array['menu']['#title'], 'Menu field should have a title');
    $this->assertArrayHasKey('#options', $form_array['menu'], 'Menu field should have options');
    $this->assertIsArray($form_array['menu']['#options'], 'Menu options should be an array');
    $this->assertArrayHasKey('#default_value', $form_array['menu'], 'Menu field should have default value');
  }

  /**
   * Tests that menu select has options from navigation architecture.
   */
  public function testMenuSelectHasOptionsFromNavigationArchitecture() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form and get menu list directly.
    $form_array = $this->form->buildForm([], $form_state);
    $menu_list = $this->navigationArchitecture->getMenuList();

    // Assert: Menu options should have 'all' as the first key.
    $this->assertArrayHasKey('all', $form_array['menu']['#options'], 'Menu options should include "all" option');

    // Assert: The 'all' option value should be a TranslatableMarkup with "All menus" text.
    $all_option = $form_array['menu']['#options']['all'];
    $this->assertInstanceOf(
      TranslatableMarkup::class,
      $all_option,
      'All menus option should be translatable'
    );

    // Assert: All menu list items should be present in options.
    foreach ($menu_list as $menu_id => $menu_name) {
      $this->assertArrayHasKey(
        $menu_id,
        $form_array['menu']['#options'],
        "Menu options should include menu: {$menu_id}"
      );
    }

    // Assert: Total count should be menu list + 1 for 'all'.
    $this->assertCount(
      count($menu_list) + 1,
      $form_array['menu']['#options'],
      'Menu options should have all menus plus the "all" option'
    );
  }

  /**
   * Tests that show parent reference checkbox has correct structure.
   */
  public function testShowParentReferenceCheckboxStructure() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Show parent reference field has correct properties.
    $this->assertEquals('checkbox', $form_array['show_parent_reference']['#type'], 'Show parent reference should be a checkbox');
    $this->assertNotEmpty($form_array['show_parent_reference']['#title'], 'Show parent reference should have a title');
    $this->assertArrayHasKey('#default_value', $form_array['show_parent_reference'], 'Show parent reference should have default value');
  }

  /**
   * Tests that show parent reference checkbox defaults to FALSE.
   */
  public function testShowParentReferenceDefaultValue() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Show parent reference defaults to FALSE when no query param.
    $this->assertEquals(
      FALSE,
      $form_array['show_parent_reference']['#default_value'],
      'Show parent reference should default to FALSE'
    );
  }

  /**
   * Tests that submit button has correct structure.
   */
  public function testSubmitButtonStructure() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Submit button has correct properties.
    $this->assertArrayHasKey('submit', $form_array['actions'], 'Form should have submit button');
    $this->assertEquals('submit', $form_array['actions']['submit']['#type'], 'Submit should be of type submit');
    $this->assertNotEmpty($form_array['actions']['submit']['#value'], 'Submit button should have a value');
  }

  /**
   * Tests form submission handling.
   */
  public function testFormSubmissionHandling() {
    // Arrange: Create form state with values.
    $form_state = new FormState();
    $form_state->setValue('menu', 'main');
    $form_state->setValue('show_parent_reference', TRUE);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Redirect URL is set.
    $redirect = $form_state->getRedirect();
    $this->assertNotNull($redirect, 'Form should set a redirect after submission');
  }

  /**
   * Tests that form submission sets correct redirect route.
   */
  public function testFormSubmissionSetsCorrectRoute() {
    // Arrange: Create form state with values.
    $form_state = new FormState();
    $form_state->setValue('menu', 'main');
    $form_state->setValue('show_parent_reference', FALSE);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Redirect URL has correct route.
    $redirect = $form_state->getRedirect();
    $this->assertEquals(
      'xray_audit.task_page.menu',
      $redirect->getRouteName(),
      'Form should redirect to menu task page route'
    );
  }

  /**
   * Tests that form submission includes menu parameter in query.
   */
  public function testFormSubmissionIncludesMenuParameter() {
    // Arrange: Create form state with menu value.
    $form_state = new FormState();
    $form_state->setValue('menu', 'main');
    $form_state->setValue('show_parent_reference', FALSE);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Query parameters include menu.
    $redirect = $form_state->getRedirect();
    $options = $redirect->getOptions();
    $this->assertArrayHasKey('query', $options, 'Redirect should have query parameters');
    $this->assertArrayHasKey('menu', $options['query'], 'Query should include menu parameter');
    $this->assertEquals('main', $options['query']['menu'], 'Menu parameter should match submitted value');
  }

  /**
   * Tests that form submission includes show_parent_reference parameter.
   */
  public function testFormSubmissionIncludesShowParentReferenceParameter() {
    // Arrange: Create form state with show_parent_reference value.
    $form_state = new FormState();
    $form_state->setValue('menu', 'main');
    $form_state->setValue('show_parent_reference', TRUE);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Query parameters include show_parent_reference.
    $redirect = $form_state->getRedirect();
    $options = $redirect->getOptions();
    $this->assertArrayHasKey('show_parent_reference', $options['query'], 'Query should include show_parent_reference parameter');
    $this->assertEquals(TRUE, $options['query']['show_parent_reference'], 'Show parent reference parameter should match submitted value');
  }

  /**
   * Tests form submission with different menu values.
   */
  public function testFormSubmissionWithDifferentMenuValues() {
    // Arrange: Get available menus.
    $menu_list = $this->navigationArchitecture->getMenuList();

    if (empty($menu_list)) {
      $this->markTestSkipped('No menus available for testing');
    }

    // Test with each available menu.
    foreach ($menu_list as $menu_id => $menu_name) {
      // Arrange: Create form state.
      $form_state = new FormState();
      $form_state->setValue('menu', $menu_id);
      $form_state->setValue('show_parent_reference', FALSE);

      // Act: Submit form.
      $form_array = [];
      $this->form->submitForm($form_array, $form_state);

      // Assert: Menu ID is in query parameters.
      $redirect = $form_state->getRedirect();
      $options = $redirect->getOptions();
      $this->assertEquals(
        $menu_id,
        $options['query']['menu'],
        "Query should contain menu ID: {$menu_id}"
      );
    }
  }

  /**
   * Tests form submission with show_parent_reference as FALSE.
   */
  public function testFormSubmissionWithShowParentReferenceFalse() {
    // Arrange: Create form state with show_parent_reference as FALSE.
    $form_state = new FormState();
    $form_state->setValue('menu', 'main');
    $form_state->setValue('show_parent_reference', FALSE);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: show_parent_reference is FALSE in query.
    $redirect = $form_state->getRedirect();
    $options = $redirect->getOptions();
    $this->assertEquals(FALSE, $options['query']['show_parent_reference'], 'Show parent reference should be FALSE');
  }

  /**
   * Tests that form builds without errors when menu list is empty.
   */
  public function testFormBuildsWithEmptyMenuList() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form (even if menu list might be empty).
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Form is still built successfully.
    $this->assertIsArray($form_array, 'Form should build as array even with empty menu list');
    $this->assertArrayHasKey('menu', $form_array, 'Form should have menu field');
    $this->assertIsArray($form_array['menu']['#options'], 'Menu options should be an array');
  }

  /**
   * Tests that form defaults to 'main' menu when available.
   */
  public function testFormDefaultsToMainMenuWhenAvailable() {
    // Arrange: Create form state and get menu list.
    $form_state = new FormState();
    $menu_list = $this->navigationArchitecture->getMenuList();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: If 'main' exists in menu list, it should be default.
    if (isset($menu_list['main'])) {
      $this->assertEquals(
        'main',
        $form_array['menu']['#default_value'],
        'Form should default to main menu when available'
      );
    }
    else {
      // If no main menu, default should be empty string.
      $this->assertEquals(
        '',
        $form_array['menu']['#default_value'],
        'Form should default to empty string when main menu not available'
      );
    }
  }

  /**
   * Tests that form handles menu field title translation.
   */
  public function testMenuFieldTitleIsTranslatable() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Menu title is a TranslatableMarkup.
    $this->assertInstanceOf(
      TranslatableMarkup::class,
      $form_array['menu']['#title'],
      'Menu field title should be translatable'
    );
  }

  /**
   * Tests that show parent reference title is translatable.
   */
  public function testShowParentReferenceTitleIsTranslatable() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Show parent reference title is a TranslatableMarkup.
    $this->assertInstanceOf(
      TranslatableMarkup::class,
      $form_array['show_parent_reference']['#title'],
      'Show parent reference title should be translatable'
    );
  }

  /**
   * Tests that submit button value is translatable.
   */
  public function testSubmitButtonValueIsTranslatable() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Submit button value is a TranslatableMarkup.
    $this->assertInstanceOf(
      TranslatableMarkup::class,
      $form_array['actions']['submit']['#value'],
      'Submit button value should be translatable'
    );
  }

  /**
   * Tests form with multiple submissions.
   */
  public function testMultipleFormSubmissions() {
    // Arrange & Act: Submit form multiple times with different values.
    $test_values = [
      ['menu' => 'main', 'show_parent_reference' => TRUE],
      ['menu' => 'footer', 'show_parent_reference' => FALSE],
      ['menu' => 'admin', 'show_parent_reference' => TRUE],
    ];

    foreach ($test_values as $values) {
      // Arrange: Create form state.
      $form_state = new FormState();
      $form_state->setValue('menu', $values['menu']);
      $form_state->setValue('show_parent_reference', $values['show_parent_reference']);

      // Act: Submit form.
      $form_array = [];
      $this->form->submitForm($form_array, $form_state);

      // Assert: Each submission sets correct redirect.
      $redirect = $form_state->getRedirect();
      $options = $redirect->getOptions();
      $this->assertEquals($values['menu'], $options['query']['menu'], "Menu should be {$values['menu']}");
      $this->assertEquals($values['show_parent_reference'], $options['query']['show_parent_reference'], "Show parent reference should be {$values['show_parent_reference']}");
    }
  }

}
